/*
   fd788_reg.h - register definitions driver for SoftLab-NSK FD788 board

   Copyright (C) 2017 - 2024 SoftLab-NSK <forward@softlab.tv>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef FD788_REG_H
#define FD788_REG_H

#include <linux/types.h>
#include <linux/io.h>

#define __read_reg(csr, addr) ioread32(&csr[addr])
#define __write_reg(csr, addr, v) iowrite32(v, &csr[addr])

typedef struct _FD788_GlobalCSR {
	uint32_t videoReset : 1;
	uint32_t vpllReset : 1;
	uint32_t reconfigure : 1;
	uint32_t vpllClkSelect : 1;
	uint32_t icapRW : 1;
	uint32_t reboot : 1;
	uint32_t _unused1506 : 10;
	uint32_t vpllLocked : 1;
	uint32_t qpll0Locked : 1;
	uint32_t qpll1Locked : 1;
	uint32_t vpllClk1Bad : 1;
	uint32_t vpllClk2Bad : 1;
	uint32_t _unused3121 : 11;
} __attribute__((packed)) FD788_GlobalCSR;

#define FD788_GlobalCSR_A (0xC0)

__always_inline FD788_GlobalCSR FD788_GlobalCSR_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_GlobalCSR_A);
	return *((FD788_GlobalCSR *)&r);
}

__always_inline void FD788_GlobalCSR_W(uint32_t *csr, FD788_GlobalCSR v)
{
	__write_reg(csr, FD788_GlobalCSR_A, *((uint32_t *)&v));
}

typedef struct _FD788_TemperatureMonitor {
	uint32_t temperature : 10;
	uint32_t maxTemperature : 10;
	uint32_t minTemperature : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD788_TemperatureMonitor;

#define FD788_TemperatureMonitor_A (0xC2)

__always_inline FD788_TemperatureMonitor FD788_TemperatureMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_TemperatureMonitor_A);
	return *((FD788_TemperatureMonitor *)&r);
}

typedef struct _FD788_VCoreMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD788_VCoreMonitor;

#define FD788_VCoreMonitor_A (0xC3)

__always_inline FD788_VCoreMonitor FD788_VCoreMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_VCoreMonitor_A);
	return *((FD788_VCoreMonitor *)&r);
}

typedef struct _FD788_VAuxMonitor {
	uint32_t voltage : 10;
	uint32_t maxVoltage : 10;
	uint32_t minVoltage : 10;
	uint32_t _unused3130 : 2;
} __attribute__((packed)) FD788_VAuxMonitor;

#define FD788_VAuxMonitor_A (0xC4)

__always_inline FD788_VAuxMonitor FD788_VAuxMonitor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_VAuxMonitor_A);
	return *((FD788_VAuxMonitor *)&r);
}

typedef struct _FD788_I2C {
	uint32_t data : 8;
	uint32_t subAddress : 8;
	uint32_t rw : 1;
	uint32_t address : 7;
	uint32_t cs : 1;
	uint32_t nack : 1;
	uint32_t _unused3126 : 6;
} __attribute__((packed)) FD788_I2C;

#define FD788_I2C_A (0xC6)

__always_inline FD788_I2C FD788_I2C_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_I2C_A);
	return *((FD788_I2C *)&r);
}

__always_inline void FD788_I2C_W(uint32_t *csr, FD788_I2C v)
{
	__write_reg(csr, FD788_I2C_A, *((uint32_t *)&v));
}

typedef struct _FD788_VideoClockCounter {
	uint32_t counter : 32;
} __attribute__((packed)) FD788_VideoClockCounter;

#define FD788_VideoClockCounter_A (0xC9)

__always_inline FD788_VideoClockCounter FD788_VideoClockCounter_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_VideoClockCounter_A);
	return *((FD788_VideoClockCounter *)&r);
}

typedef struct _FD788_ClockFrequency {
	uint32_t frequency : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD788_ClockFrequency;

#define FD788_ClockFrequency_A (0xCA)

__always_inline FD788_ClockFrequency FD788_ClockFrequency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_ClockFrequency_A);
	return *((FD788_ClockFrequency *)&r);
}

typedef struct _FD788_ClockDeviation {
	uint32_t deviation : 24;
	uint32_t _unused3024 : 7;
	uint32_t direction : 1;
} __attribute__((packed)) FD788_ClockDeviation;

#define FD788_ClockDeviation_A (0xCB)

__always_inline FD788_ClockDeviation FD788_ClockDeviation_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_ClockDeviation_A);
	return *((FD788_ClockDeviation *)&r);
}

__always_inline void FD788_ClockDeviation_W(uint32_t *csr, FD788_ClockDeviation v)
{
	__write_reg(csr, FD788_ClockDeviation_A, *((uint32_t *)&v));
}

typedef struct _FD788_VideoConfig {
	uint32_t io0 : 1;
	uint32_t io1 : 1;
	uint32_t io2 : 1;
	uint32_t io3 : 1;
	uint32_t io4 : 1;
	uint32_t io5 : 1;
	uint32_t io6 : 1;
	uint32_t io7 : 1;
	uint32_t _unused3108 : 24;
} __attribute__((packed)) FD788_VideoConfig;

#define FD788_VideoConfig_A (0xCC)

__always_inline FD788_VideoConfig FD788_VideoConfig_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_VideoConfig_A);
	return *((FD788_VideoConfig *)&r);
}

__always_inline void FD788_VideoConfig_W(uint32_t *csr, FD788_VideoConfig v)
{
	__write_reg(csr, FD788_VideoConfig_A, *((uint32_t *)&v));
}

typedef struct _FD788_SoftID {
	uint32_t id : 32;
} __attribute__((packed)) FD788_SoftID;

#define FD788_SoftID_A (0xCF)

__always_inline FD788_SoftID FD788_SoftID_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_SoftID_A);
	return *((FD788_SoftID *)&r);
}

typedef struct _FD788_SPIControl {
	uint32_t txBits : 12;
	uint32_t rxBits : 12;
	uint32_t bus : 8;
} __attribute__((packed)) FD788_SPIControl;

#define FD788_SPIControl_A (0xD2)

__always_inline void FD788_SPIControl_W(uint32_t *csr, FD788_SPIControl v)
{
	__write_reg(csr, FD788_SPIControl_A, *((uint32_t *)&v));
}

typedef struct _FD788_SPIData {
	uint32_t data : 32;
} __attribute__((packed)) FD788_SPIData;

#define FD788_SPIData_A (0xD3)

__always_inline FD788_SPIData FD788_SPIData_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_SPIData_A);
	return *((FD788_SPIData *)&r);
}

__always_inline void FD788_SPIData_W(uint32_t *csr, FD788_SPIData v)
{
	__write_reg(csr, FD788_SPIData_A, *((uint32_t *)&v));
}

typedef struct _FD788_StaticVersion {
	uint32_t revision : 16;
	uint32_t minor : 8;
	uint32_t major : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD788_StaticVersion;

#define FD788_StaticVersion_A (0xFF)

__always_inline FD788_StaticVersion FD788_StaticVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_StaticVersion_A);
	return *((FD788_StaticVersion *)&r);
}

__always_inline void FD788_StaticVersion_W(uint32_t *csr, FD788_StaticVersion v)
{
	__write_reg(csr, 0xFF, *((uint32_t *)&v));
}

typedef struct _FD788_VDMA {
	uint32_t _unused3000 : 31;
	uint32_t enable : 1;
} __attribute__((packed)) FD788_VDMA;

#define FD788_VDMA_A (0x00)

__always_inline FD788_VDMA FD788_VDMA_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_VDMA_A);
	return *((FD788_VDMA *)&r);
}

__always_inline void FD788_VDMA_W(uint32_t *csr, FD788_VDMA v)
{
	__write_reg(csr, FD788_VDMA_A, *((uint32_t *)&v));
}

typedef struct _FD788_VDMADescriptor {
	uint32_t address : 32;
} __attribute__((packed)) FD788_VDMADescriptor;

#define FD788_VDMADescriptor_A (0x01)

__always_inline FD788_VDMADescriptor FD788_VDMADescriptor_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_VDMADescriptor_A);
	return *((FD788_VDMADescriptor *)&r);
}

__always_inline void FD788_VDMADescriptor_W(uint32_t *csr, FD788_VDMADescriptor v)
{
	__write_reg(csr, FD788_VDMADescriptor_A, *((uint32_t *)&v));
}

typedef struct _FD788_IRQEnable {
	uint32_t io0IRQ : 1;
	uint32_t _unused0201 : 2;
	uint32_t io1IRQ : 1;
	uint32_t _unused0504 : 2;
	uint32_t io2IRQ : 1;
	uint32_t _unused0807 : 2;
	uint32_t io3IRQ : 1;
	uint32_t _unused1110 : 2;
	uint32_t io4IRQ : 1;
	uint32_t _unused1413 : 2;
	uint32_t io5IRQ : 1;
	uint32_t _unused1716 : 2;
	uint32_t io6IRQ : 1;
	uint32_t _unused2019 : 2;
	uint32_t io7IRQ : 1;
	uint32_t _unused2322 : 2;
	uint32_t sync0IRQ : 1;
	uint32_t _unused2525 : 1;
	uint32_t sync1IRQ : 1;
	uint32_t _unused2727 : 1;
	uint32_t sync2IRQ : 1;
	uint32_t _unused2929 : 1;
	uint32_t sync3IRQ : 1;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD788_IRQEnable;

#define FD788_IRQEnable_A (0x04)

__always_inline FD788_IRQEnable FD788_IRQEnable_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_IRQEnable_A);
	return *((FD788_IRQEnable *)&r);
}

__always_inline void FD788_IRQEnable_W(uint32_t *csr, FD788_IRQEnable v)
{
	__write_reg(csr, FD788_IRQEnable_A, *((uint32_t *)&v));
}

typedef struct _FD788_IRQFlags {
	uint32_t io0IRQ : 1;
	uint32_t io0Buffer : 2;
	uint32_t io1IRQ : 1;
	uint32_t io1Buffer : 2;
	uint32_t io2IRQ : 1;
	uint32_t io2Buffer : 2;
	uint32_t io3IRQ : 1;
	uint32_t io3Buffer : 2;
	uint32_t io4IRQ : 1;
	uint32_t io4Buffer : 2;
	uint32_t io5IRQ : 1;
	uint32_t io5Buffer : 2;
	uint32_t io6IRQ : 1;
	uint32_t io6Buffer : 2;
	uint32_t io7IRQ : 1;
	uint32_t io7Buffer : 2;
	uint32_t sync0IRQ : 1;
	uint32_t sync0Field : 1;
	uint32_t sync1IRQ : 1;
	uint32_t sync1Field : 1;
	uint32_t sync2IRQ : 1;
	uint32_t sync2Field : 1;
	uint32_t sync3IRQ : 1;
	uint32_t sync3Field : 1;
} __attribute__((packed)) FD788_IRQFlags;

#define FD788_IRQFlags_A (0x05)

__always_inline FD788_IRQFlags FD788_IRQFlags_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_IRQFlags_A);
	return *((FD788_IRQFlags *)&r);
}

__always_inline void FD788_IRQFlags_W(uint32_t *csr, FD788_IRQFlags v)
{
	__write_reg(csr, FD788_IRQFlags_A, *((uint32_t *)&v));
}

typedef struct _FD788_AnalogSync {
	uint32_t hsync : 11;
	uint32_t vsync : 20;
	uint32_t _unused3131 : 1;
} __attribute__((packed)) FD788_AnalogSync;

#define FD788_AnalogSync_A (0x06)

__always_inline FD788_AnalogSync FD788_AnalogSync_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_AnalogSync_A);
	return *((FD788_AnalogSync *)&r);
}

typedef struct _FD788_Genlock {
	uint32_t mode : 2;
	uint32_t lock : 1;
	uint32_t _unused0803 : 6;
	uint32_t offset : 23;
} __attribute__((packed)) FD788_Genlock;

#define FD788_Genlock_A (0x07)

__always_inline FD788_Genlock FD788_Genlock_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_Genlock_A);
	return *((FD788_Genlock *)&r);
}

__always_inline void FD788_Genlock_W(uint32_t *csr, FD788_Genlock v)
{
	__write_reg(csr, FD788_Genlock_A, *((uint32_t *)&v));
}

typedef struct _FD788_SyncGenerator {
	uint32_t frameSize : 23;
	uint32_t _unused3023 : 8;
	uint32_t sync : 1;
} __attribute__((packed)) FD788_SyncGenerator;

#define FD788_SyncGenerator_A(n) (0x08 + n * 4)

__always_inline FD788_SyncGenerator FD788_SyncGenerator_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_SyncGenerator_A(n));
	return *((FD788_SyncGenerator *)&r);
}

__always_inline void FD788_SyncGenerator_W(uint32_t *csr, unsigned int n, FD788_SyncGenerator v)
{
	__write_reg(csr, FD788_SyncGenerator_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_SyncGeneratorField {
	uint32_t fieldSize : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD788_SyncGeneratorField;

#define FD788_SyncGeneratorField_A(n) (0x09 + n * 4)

__always_inline FD788_SyncGeneratorField FD788_SyncGeneratorField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_SyncGeneratorField_A(n));
	return *((FD788_SyncGeneratorField *)&r);
}

__always_inline void FD788_SyncGeneratorField_W(uint32_t *csr, unsigned int n,
						FD788_SyncGeneratorField v)
{
	__write_reg(csr, FD788_SyncGeneratorField_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_SyncGeneratorTime {
	uint32_t time : 32;
} __attribute__((packed)) FD788_SyncGeneratorTime;

#define FD788_SyncGeneratorTime_A(n) (0x0A + n * 4)

__always_inline FD788_SyncGeneratorTime FD788_SyncGeneratorTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_SyncGeneratorTime_A(n));
	return *((FD788_SyncGeneratorTime *)&r);
}

typedef struct _FD788_SyncGeneratorPhase {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD788_SyncGeneratorPhase;

#define FD788_SyncGeneratorPhase_A(n) (0x0B + n * 4)

__always_inline void FD788_SyncGeneratorPhase_W(uint32_t *csr, unsigned int n,
						FD788_SyncGeneratorPhase v)
{
	__write_reg(csr, FD788_SyncGeneratorPhase_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_PCIEPerfomance {
	uint32_t txCounter : 16;
	uint32_t rxCounter : 16;
} __attribute__((packed)) FD788_PCIEPerfomance;

#define FD788_PCIEPerfomance_A (0x18)

__always_inline FD788_PCIEPerfomance FD788_PCIEPerfomance_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PCIEPerfomance_A);
	return *((FD788_PCIEPerfomance *)&r);
}

typedef struct _FD788_Video4KSwitch {
	uint32_t qi0 : 2;
	uint32_t qi1 : 2;
	uint32_t qi2 : 2;
	uint32_t qi3 : 2;
	uint32_t di0 : 2;
	uint32_t di1 : 2;
	uint32_t _unused1512 : 4;
	uint32_t qo0 : 2;
	uint32_t qo1 : 2;
	uint32_t qo2 : 2;
	uint32_t qo3 : 2;
	uint32_t do0 : 2;
	uint32_t do1 : 2;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD788_Video4KSwitch;

#define FD788_Video4KSwitch_A (0x19)

__always_inline FD788_Video4KSwitch FD788_Video4KSwitch_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_Video4KSwitch_A);
	return *((FD788_Video4KSwitch *)&r);
}

__always_inline void FD788_Video4KSwitch_W(uint32_t *csr, FD788_Video4KSwitch v)
{
	__write_reg(csr, FD788_Video4KSwitch_A, *((uint32_t *)&v));
}

typedef struct _FD788_PCIERxMaxLatency {
	uint32_t maxLatency : 16;
	uint32_t numRequests : 16;
} __attribute__((packed)) FD788_PCIERxMaxLatency;

#define FD788_PCIERxMaxLatency_A (0x1B)

__always_inline FD788_PCIERxMaxLatency FD788_PCIERxMaxLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PCIERxMaxLatency_A);
	return *((FD788_PCIERxMaxLatency *)&r);
}

typedef struct _FD788_PCIERxTotalLatency {
	uint32_t latency : 32;
} __attribute__((packed)) FD788_PCIERxTotalLatency;

#define FD788_PCIERxTotalLatency_A (0x1C)

__always_inline FD788_PCIERxTotalLatency FD788_PCIERxTotalLatency_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PCIERxTotalLatency_A);
	return *((FD788_PCIERxTotalLatency *)&r);
}

typedef struct _FD788_AnalogVSyncTime {
	uint32_t time : 32;
} __attribute__((packed)) FD788_AnalogVSyncTime;

#define FD788_AnalogVSyncTime_A (0x1D)

__always_inline FD788_AnalogVSyncTime FD788_AnalogVSyncTime_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_AnalogVSyncTime_A);
	return *((FD788_AnalogVSyncTime *)&r);
}

typedef struct _FD788_DynamicVersion {
	uint32_t revision : 16;
	uint32_t minor : 8;
	uint32_t hardware : 4;
	uint32_t project : 4;
} __attribute__((packed)) FD788_DynamicVersion;

#define FD788_DynamicVersion_A (0x3F)

__always_inline FD788_DynamicVersion FD788_DynamicVersion_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_DynamicVersion_A);
	return *((FD788_DynamicVersion *)&r);
}

typedef struct _FD788_VideoInCS {
	uint32_t cd : 1;
	uint32_t modeLocked : 1;
	uint32_t levelB : 1;
	uint32_t mode : 2;
	uint32_t reset : 1;
	uint32_t formatLocked : 1;
	uint32_t progressive : 1;
	uint32_t forceMode : 1;
	uint32_t _unused0909 : 1;
	uint32_t capture : 1;
	uint32_t captureRaw : 1;
	uint32_t vbiTop : 1;
	uint32_t vbiBottom : 1;
	uint32_t vis10b : 1;
	uint32_t vbi10b : 1;
	uint32_t ycSwap : 1;
	uint32_t asiPeriod : 2;
	uint32_t numStreams : 2;
	uint32_t _unused2421 : 4;
	uint32_t mode2 : 1;
	uint32_t vancCapture : 1;
	uint32_t dqLocked : 1;
	uint32_t dqQuad : 1;
	uint32_t dqEnable : 1;
	uint32_t _unused3030 : 1;
	uint32_t test : 1;
} __attribute__((packed)) FD788_VideoInCS;

#define FD788_VideoInCS_A(n) (0x40 + n * 16)

__always_inline FD788_VideoInCS FD788_VideoInCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInCS_A(n));
	return *((FD788_VideoInCS *)&r);
}

__always_inline void FD788_VideoInCS_W(uint32_t *csr, unsigned int n, FD788_VideoInCS v)
{
	__write_reg(csr, FD788_VideoInCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoInLine {
	uint32_t totalLines : 12;
	uint32_t _unused1512 : 4;
	uint32_t activeLines : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD788_VideoInLine;

#define FD788_VideoInLine_A(n) (0x41 + n * 16)

__always_inline FD788_VideoInLine FD788_VideoInLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInLine_A(n));
	return *((FD788_VideoInLine *)&r);
}

typedef struct _FD788_VideoInPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD788_VideoInPixel;

#define FD788_VideoInPixel_A(n) (0x42 + n * 16)

__always_inline FD788_VideoInPixel FD788_VideoInPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInPixel_A(n));
	return *((FD788_VideoInPixel *)&r);
}

typedef struct _FD788_VideoInIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD788_VideoInIRQRefTime;

#define FD788_VideoInIRQRefTime_A(n) (0x44 + n * 16)

__always_inline FD788_VideoInIRQRefTime FD788_VideoInIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInIRQRefTime_A(n));
	return *((FD788_VideoInIRQRefTime *)&r);
}

typedef struct _FD788_VideoInANCCounter {
	uint32_t counter : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD788_VideoInANCCounter;

#define FD788_VideoInANCCounter_A(n) (0x46 + n * 16)

__always_inline FD788_VideoInANCCounter FD788_VideoInANCCounter_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInANCCounter_A(n));
	return *((FD788_VideoInANCCounter *)&r);
}

typedef struct _FD788_VideoInVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD788_VideoInVPID;

#define FD788_VideoInVPID_A(n) (0x47 + n * 16)

__always_inline FD788_VideoInVPID FD788_VideoInVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInVPID_A(n));
	return *((FD788_VideoInVPID *)&r);
}

typedef struct _FD788_VideoInFrameSize {
	uint32_t size : 21;
	uint32_t _unused3121 : 11;
} __attribute__((packed)) FD788_VideoInFrameSize;

#define FD788_VideoInFrameSize_A(n) (0x48 + n * 16)

__always_inline FD788_VideoInFrameSize FD788_VideoInFrameSize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInFrameSize_A(n));
	return *((FD788_VideoInFrameSize *)&r);
}

typedef struct _FD788_VideoInFIFOStatus {
	uint32_t videoMax : 15;
	uint32_t videoError : 1;
	uint32_t audioMax : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD788_VideoInFIFOStatus;

#define FD788_VideoInFIFOStatus_A(n) (0x4A + n * 16)

__always_inline FD788_VideoInFIFOStatus FD788_VideoInFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInFIFOStatus_A(n));
	return *((FD788_VideoInFIFOStatus *)&r);
}

typedef struct _FD788_VideoInVBISize {
	uint32_t size : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD788_VideoInVBISize;

#define FD788_VideoInVBISize_A(n) (0x4E + n * 16)

__always_inline FD788_VideoInVBISize FD788_VideoInVBISize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInVBISize_A(n));
	return *((FD788_VideoInVBISize *)&r);
}

typedef struct _FD788_VideoInVisibleSize {
	uint32_t size : 25;
	uint32_t _unused3125 : 7;
} __attribute__((packed)) FD788_VideoInVisibleSize;

#define FD788_VideoInVisibleSize_A(n) (0x4F + n * 16)

__always_inline FD788_VideoInVisibleSize FD788_VideoInVisibleSize_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoInVisibleSize_A(n));
	return *((FD788_VideoInVisibleSize *)&r);
}

typedef struct _FD788_VideoOutCS {
	uint32_t reset : 1;
	uint32_t mute : 1;
	uint32_t levelB : 1;
	uint32_t mode : 2;
	uint32_t clockM : 1;
	uint32_t freeRunning : 1;
	uint32_t progressive : 1;
	uint32_t audioCount : 2;
	uint32_t playback : 1;
	uint32_t playbackRaw : 1;
	uint32_t vbiTop : 1;
	uint32_t vbiBottom : 1;
	uint32_t vis10b : 1;
	uint32_t vbi10b : 1;
	uint32_t dataPresent : 1;
	uint32_t syncSource : 2;
	uint32_t atcEnable : 1;
	uint32_t atcType : 2;
	uint32_t dpllEnable : 1;
	uint32_t dpllSelect : 3;
	uint32_t watchdogEn : 1;
	uint32_t _unused2727 : 1;
	uint32_t dqQuad : 1;
	uint32_t dqEnable : 1;
	uint32_t clone : 1;
	uint32_t test : 1;
} __attribute__((packed)) FD788_VideoOutCS;

#define FD788_VideoOutCS_A(n) (0x40 + n * 16)

__always_inline FD788_VideoOutCS FD788_VideoOutCS_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutCS_A(n));
	return *((FD788_VideoOutCS *)&r);
}

__always_inline void FD788_VideoOutCS_W(uint32_t *csr, unsigned int n, FD788_VideoOutCS v)
{
	__write_reg(csr, FD788_VideoOutCS_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutLine {
	uint32_t totalLines : 12;
	uint32_t _unused3112 : 20;
} __attribute__((packed)) FD788_VideoOutLine;

#define FD788_VideoOutLine_A(n) (0x41 + n * 16)

__always_inline FD788_VideoOutLine FD788_VideoOutLine_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutLine_A(n));
	return *((FD788_VideoOutLine *)&r);
}

__always_inline void FD788_VideoOutLine_W(uint32_t *csr, unsigned int n, FD788_VideoOutLine v)
{
	__write_reg(csr, FD788_VideoOutLine_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutPixel {
	uint32_t totalPixels : 13;
	uint32_t _unused1513 : 3;
	uint32_t activePixels : 13;
	uint32_t _unused3129 : 3;
} __attribute__((packed)) FD788_VideoOutPixel;

#define FD788_VideoOutPixel_A(n) (0x42 + n * 16)

__always_inline FD788_VideoOutPixel FD788_VideoOutPixel_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutPixel_A(n));
	return *((FD788_VideoOutPixel *)&r);
}

__always_inline void FD788_VideoOutPixel_W(uint32_t *csr, unsigned int n, FD788_VideoOutPixel v)
{
	__write_reg(csr, FD788_VideoOutPixel_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutStart {
	uint32_t startOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t startEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD788_VideoOutStart;

#define FD788_VideoOutStart_A(n) (0x43 + n * 16)

__always_inline FD788_VideoOutStart FD788_VideoOutStart_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutStart_A(n));
	return *((FD788_VideoOutStart *)&r);
}

__always_inline void FD788_VideoOutStart_W(uint32_t *csr, unsigned int n, FD788_VideoOutStart v)
{
	__write_reg(csr, FD788_VideoOutStart_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutStop {
	uint32_t stopOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t stopEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD788_VideoOutStop;

#define FD788_VideoOutStop_A(n) (0x44 + n * 16)

__always_inline FD788_VideoOutStop FD788_VideoOutStop_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutStop_A(n));
	return *((FD788_VideoOutStop *)&r);
}

__always_inline void FD788_VideoOutStop_W(uint32_t *csr, unsigned int n, FD788_VideoOutStop v)
{
	__write_reg(csr, FD788_VideoOutStop_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutField {
	uint32_t switchOdd : 12;
	uint32_t _unused1512 : 4;
	uint32_t switchEven : 12;
	uint32_t _unused3128 : 4;
} __attribute__((packed)) FD788_VideoOutField;

#define FD788_VideoOutField_A(n) (0x45 + n * 16)

__always_inline FD788_VideoOutField FD788_VideoOutField_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutField_A(n));
	return *((FD788_VideoOutField *)&r);
}

__always_inline void FD788_VideoOutField_W(uint32_t *csr, unsigned int n, FD788_VideoOutField v)
{
	__write_reg(csr, FD788_VideoOutField_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutAudio {
	uint32_t address : 20;
	uint32_t _unused3120 : 12;
} __attribute__((packed)) FD788_VideoOutAudio;

#define FD788_VideoOutAudio_A(n) (0x46 + n * 16)

__always_inline FD788_VideoOutAudio FD788_VideoOutAudio_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutAudio_A(n));
	return *((FD788_VideoOutAudio *)&r);
}

__always_inline void FD788_VideoOutAudio_W(uint32_t *csr, unsigned int n, FD788_VideoOutAudio v)
{
	__write_reg(csr, FD788_VideoOutAudio_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutVPID {
	uint32_t vpid : 32;
} __attribute__((packed)) FD788_VideoOutVPID;

#define FD788_VideoOutVPID_A(n) (0x47 + n * 16)

__always_inline FD788_VideoOutVPID FD788_VideoOutVPID_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutVPID_A(n));
	return *((FD788_VideoOutVPID *)&r);
}

__always_inline void FD788_VideoOutVPID_W(uint32_t *csr, unsigned int n, FD788_VideoOutVPID v)
{
	__write_reg(csr, FD788_VideoOutVPID_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutOddDataCount {
	uint32_t count : 24;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD788_VideoOutOddDataCount;

#define FD788_VideoOutOddDataCount_A(n) (0x48 + n * 16)

__always_inline FD788_VideoOutOddDataCount FD788_VideoOutOddDataCount_R(uint32_t *csr,
									unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutOddDataCount_A(n));
	return *((FD788_VideoOutOddDataCount *)&r);
}

__always_inline void FD788_VideoOutOddDataCount_W(uint32_t *csr, unsigned int n,
						  FD788_VideoOutOddDataCount v)
{
	__write_reg(csr, FD788_VideoOutOddDataCount_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutEvenDataCount {
	uint32_t count : 24;
	uint32_t _unused3124 : 8;
} __attribute__((packed)) FD788_VideoOutEvenDataCount;

#define FD788_VideoOutEvenDataCount_A(n) (0x49 + n * 16)

__always_inline FD788_VideoOutEvenDataCount FD788_VideoOutEvenDataCount_R(uint32_t *csr,
									  unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutEvenDataCount_A(n));
	return *((FD788_VideoOutEvenDataCount *)&r);
}

__always_inline void FD788_VideoOutEvenDataCount_W(uint32_t *csr, unsigned int n,
						   FD788_VideoOutEvenDataCount v)
{
	__write_reg(csr, FD788_VideoOutEvenDataCount_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutFIFOStatus {
	uint32_t videoMin : 15;
	uint32_t videoError : 1;
	uint32_t audioMin : 15;
	uint32_t audioError : 1;
} __attribute__((packed)) FD788_VideoOutFIFOStatus;

#define FD788_VideoOutFIFOStatus_A(n) (0x4A + n * 16)

__always_inline FD788_VideoOutFIFOStatus FD788_VideoOutFIFOStatus_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutFIFOStatus_A(n));
	return *((FD788_VideoOutFIFOStatus *)&r);
}

#define FD788_VideoOutATC_A(n) (0x4B + n * 16)

typedef struct _FD788_VideoOutIRQRefTime {
	uint32_t time : 32;
} __attribute__((packed)) FD788_VideoOutIRQRefTime;

#define FD788_VideoOutIRQRefTime_A(n) (0x4C + n * 16)

__always_inline FD788_VideoOutIRQRefTime FD788_VideoOutIRQRefTime_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutIRQRefTime_A(n));
	return *((FD788_VideoOutIRQRefTime *)&r);
}

typedef struct _FD788_VideoOutPhaseShift {
	uint32_t phase : 23;
	uint32_t _unused3123 : 9;
} __attribute__((packed)) FD788_VideoOutPhaseShift;

#define FD788_VideoOutPhaseShift_A(n) (0x4D + n * 16)

__always_inline FD788_VideoOutPhaseShift FD788_VideoOutPhaseShift_R(uint32_t *csr, unsigned int n)
{
	uint32_t r = __read_reg(csr, FD788_VideoOutPhaseShift_A(n));
	return *((FD788_VideoOutPhaseShift *)&r);
}

__always_inline void FD788_VideoOutPhaseShift_W(uint32_t *csr, unsigned int n,
						FD788_VideoOutPhaseShift v)
{
	__write_reg(csr, FD788_VideoOutPhaseShift_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_VideoOutWatchdog {
	uint32_t timeout : 16;
	uint32_t _unused3116 : 16;
} __attribute__((packed)) FD788_VideoOutWatchdog;

#define FD788_VideoOutWatchdog_A(n) (0x4E + n * 16)

__always_inline void FD788_VideoOutWatchdog_W(uint32_t *csr, unsigned int n,
					      FD788_VideoOutWatchdog v)
{
	__write_reg(csr, FD788_VideoOutWatchdog_A(n), *((uint32_t *)&v));
}

typedef struct _FD788_PPSLTCControl {
	uint32_t mode : 2;
	uint32_t _unused1502 : 14;
	uint32_t valid : 1;
	uint32_t _unused3117 : 15;
} __attribute__((packed)) FD788_PPSLTCControl;

#define FD788_PPSLTCControl_A (0x1E)

__always_inline FD788_PPSLTCControl FD788_PPSLTCControl_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PPSLTCControl_A);
	return *((FD788_PPSLTCControl *)&r);
}

__always_inline void FD788_PPSLTCControl_W(uint32_t *csr, FD788_PPSLTCControl v)
{
	__write_reg(csr, FD788_PPSLTCControl_A, *((uint32_t *)&v));
}

typedef struct _FD788_PPSLTCTimestamp {
	uint32_t time : 32;
} __attribute__((packed)) FD788_PPSLTCTimestamp;

#define FD788_PPSLTCTimestamp_A (0x1F)

__always_inline FD788_PPSLTCTimestamp FD788_PPSLTCTimestamp_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PPSLTCTimestamp_A);
	return *((FD788_PPSLTCTimestamp *)&r);
}

typedef struct _FD788_PPSLTCDataLow {
	uint32_t data : 32;
} __attribute__((packed)) FD788_PPSLTCDataLow;

#define FD788_PPSLTCDataLow_A (0x20)

__always_inline FD788_PPSLTCDataLow FD788_PPSLTCDataLow_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PPSLTCDataLow_A);
	return *((FD788_PPSLTCDataLow *)&r);
}

typedef struct _FD788_PPSLTCDataHigh {
	uint32_t data : 32;
} __attribute__((packed)) FD788_PPSLTCDataHigh;

#define FD788_PPSLTCDataHigh_A (0x21)

__always_inline FD788_PPSLTCDataHigh FD788_PPSLTCDataHigh_R(uint32_t *csr)
{
	uint32_t r = __read_reg(csr, FD788_PPSLTCDataHigh_A);
	return *((FD788_PPSLTCDataHigh *)&r);
}

#endif
