#include "ALSADevice.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"

#include <QApplication>
#include <QImage>
#include <QPainter>
#include <QWidget>

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>
#include <thread>

/*!
 * \dir ./
 * This example shows A/V delay using Qt5
 */

namespace {
static const int BUFFER_COUNT = 8;
static const int TEST_PERIOD_FIELDS = 20;
// ALSA chunk size, should be less than frame (field) period
static const int AUDIO_PERIOD_SIZE_US = 10000;
static const int AUDIO_BUFFER_SIZE_US = 1000000;
static const int AUDIO_CHANNELS = 1;
static const snd_pcm_uframes_t AUDIO_SAMPLE_RATE = 48000;
static const snd_pcm_format_t AUDIO_FORMAT = SND_PCM_FORMAT_S32;
static const snd_pcm_uframes_t AUDIO_BUFFER_SIZE = 2048 * 16;
static const int MAX_AUDIO_PER_FRAME = 2048;
}

static bool shouldStop = false;

// Qt Window
class AVDelayWindow : public QWidget {
    Q_OBJECT
public:
    AVDelayWindow(QWidget* parent = nullptr)
    {
        m_video = QImage(TEST_PERIOD_FIELDS, 1, QImage::Format_Grayscale8);
        m_audio = QImage(TEST_PERIOD_FIELDS, MAX_AUDIO_PER_FRAME, QImage::Format_Grayscale8);
        m_video.fill(0);
        m_audio.fill(0);
        m_audioPerFrame = std::vector<int>(TEST_PERIOD_FIELDS, 0);
        m_curPosition = 0;
    }

public:
    void newFrame(uint8_t videoValue, int32_t* audioValue, int audioSize)
    {
        // if ((videoValue <= 16))
        //     m_curPosition = 0;

        m_video.bits()[m_curPosition] = videoValue;
        for (int i = 0; i < std::min(audioSize, MAX_AUDIO_PER_FRAME); i++) {
            int v = abs(audioValue[i] >> 23);
            m_audio.scanLine(i)[m_curPosition] = (v > 255 ? 255 : v);
        }

        m_audioPerFrame[m_curPosition] = std::min(audioSize, MAX_AUDIO_PER_FRAME);

        m_curPosition = (m_curPosition + 1) % TEST_PERIOD_FIELDS;
        update();
    }

protected:
    void paintEvent(QPaintEvent* event) override
    {
        QPainter p(this);
        int w = width(), h = height();
        QPen timePen(Qt::green, 0), framePen(Qt::blue, 0), borderPen(Qt::red, 0);

        p.setRenderHint(QPainter::Antialiasing, false);

        p.drawImage(QRect(0, 0, w, h / 10), m_video);
        p.drawImage(QRect(0, h / 10, w, h * 9 / 10), m_audio);

        p.setPen(framePen);
        for (int f = 0; f < TEST_PERIOD_FIELDS; f++) {
            int vx = f * w / TEST_PERIOD_FIELDS;
            p.drawLine(vx, 0, vx, h);
        }

        p.setPen(timePen);
        for (int t = 0; t < MAX_AUDIO_PER_FRAME; t += 96) {
            int vy = t * h * 9 / 10 / MAX_AUDIO_PER_FRAME + h / 10;
            p.drawLine(0, vy, w, vy);
        }

        p.setPen(borderPen);
        p.drawLine(0, h / 10, w, h / 10);
        for (int f = 0; f < TEST_PERIOD_FIELDS; f++) {
            int vy = m_audioPerFrame[f] * h * 9 / 10 / MAX_AUDIO_PER_FRAME + h / 10;
            int vx1 = f * w / TEST_PERIOD_FIELDS;
            int vx2 = (f + 1) * w / TEST_PERIOD_FIELDS;
            p.drawLine(vx1, vy, vx2, vy);
        }
    }
    void closeEvent(QCloseEvent* event) override
    {
        shouldStop = true;
    }

private:
    QImage m_audio, m_video;
    int m_curPosition;
    std::vector<int> m_audioPerFrame;
};

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
    qApp->quit();
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt << std::endl;

    return fmt;
}

void setupAudio(ALSADevice& dev)
{
    snd_pcm_hw_params_t* params = nullptr;
    snd_pcm_sw_params_t* sw_params = nullptr;
    int err = 0;
    snd_pcm_format_t format;
    snd_pcm_uframes_t period;
    unsigned int channels, rate;
    int dir = 0;

    // Alloc hardware parameter structs
    snd_pcm_hw_params_malloc(&params);

    // Get default hardware parameters
    snd_pcm_hw_params_any(dev.pcm().get(), params);

    // Interleave channels
    snd_pcm_hw_params_set_access(dev.pcm().get(), params, SND_PCM_ACCESS_RW_INTERLEAVED);

    // Set format, number of channels, sample rate
    snd_pcm_hw_params_set_channels(dev.pcm().get(), params, AUDIO_CHANNELS);
    snd_pcm_hw_params_set_rate(dev.pcm().get(), params, AUDIO_SAMPLE_RATE, 0);
    snd_pcm_hw_params_set_format(dev.pcm().get(), params, AUDIO_FORMAT);

    // Set period and buffer sizes
    snd_pcm_hw_params_set_period_time(dev.pcm().get(), params, AUDIO_PERIOD_SIZE_US, 0);
    snd_pcm_hw_params_set_buffer_time(dev.pcm().get(), params, AUDIO_BUFFER_SIZE_US, 0);

    // Set hardware parameters, you can also change format, channels and other params prior this
    // call
    snd_pcm_hw_params(dev.pcm().get(), params);

    snd_pcm_hw_params_free(params);

    // Extract accepted parameters,
    snd_pcm_hw_params_get_format(params, &format);
    snd_pcm_hw_params_get_channels(params, &channels);
    snd_pcm_hw_params_get_rate(params, &rate, &dir);
    snd_pcm_hw_params_get_period_size(params, &period, &dir);

    // Alloc software parameter structs
    snd_pcm_sw_params_alloca(&sw_params);
    // Get current software parameters
    snd_pcm_sw_params_current(dev.pcm().get(), sw_params);
    // Enable ALSA timestamping so we can always sync audio with video
    snd_pcm_sw_params_set_tstamp_mode(dev.pcm().get(), sw_params, SND_PCM_TSTAMP_ENABLE);
    // Disable ALSA auto-start
    snd_pcm_sw_params_set_start_threshold(dev.pcm().get(), sw_params, LONG_MAX);
    // Set software parameters
    snd_pcm_sw_params(dev.pcm().get(), sw_params);

    snd_pcm_prepare(dev.pcm().get());

    std::cout << "ALSA capture " << channels << " x " << snd_pcm_format_name(format) << " x "
              << rate << " with period of " << period << " samples" << std::endl;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers,
    const v4l2_pix_format_mplane& fmt)
{
    // Fill capture queue
    for (auto buf : buffers) {
        dev.queueBuffer(buf);
    }
}

void mainLoop(
    V4L2Device& dev, ALSADevice& audioDev, const v4l2_pix_format_mplane& fmt, AVDelayWindow* window)
{
    int type = dev.type();
    int32_t* audio_buf = new int32_t[AUDIO_BUFFER_SIZE * AUDIO_CHANNELS];
    std::shared_ptr<snd_pcm_status_t> status;
    snd_pcm_status_t* sp;
    snd_pcm_status_malloc(&sp);
    status = std::shared_ptr<snd_pcm_status_t>(sp, snd_pcm_status_free);

    // Start capture
    snd_pcm_start(audioDev.pcm().get());
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLIN;

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLIN))
            continue;

        // Get buffer from queue
        auto videoBuf = dev.dequeueBuffer();

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Received buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << videoBuf->v4l2buf().index
                  << ", sequence = " << videoBuf->v4l2buf().sequence
                  << ", field = " << videoBuf->v4l2buf().field << ", timestamp = "
                  << (uint64_t)videoBuf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + videoBuf->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, size = " << videoBuf->bytesused(0) << std::endl;

        snd_pcm_status(audioDev.pcm().get(), status.get());
        snd_pcm_uframes_t avail = snd_pcm_status_get_avail(status.get());
        snd_pcm_sframes_t size
            = snd_pcm_readi(audioDev.pcm().get(), audio_buf, std::min(avail, AUDIO_BUFFER_SIZE));

        std::cout << "Received " << size << " audio frames" << std::endl;

        // Pass it to window
        if (fmt.field != V4L2_FIELD_INTERLACED) {
            QMetaObject::invokeMethod(
                window, [=] { window->newFrame(videoBuf->data(0)[1], audio_buf, size); },
                Qt::QueuedConnection);
        } else {
            QMetaObject::invokeMethod(
                window, [=] { window->newFrame(videoBuf->data(0)[1], audio_buf, size / 2); },
                Qt::QueuedConnection);
            QMetaObject::invokeMethod(
                window,
                [=] {
                    window->newFrame(
                        videoBuf->data(0)[fmt.width * 2 + 1], audio_buf + size / 2, size / 2);
                },
                Qt::QueuedConnection);
        }

        // Return buffer back to queue
        dev.queueBuffer(videoBuf);
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int videoThread(std::string videoPath, std::string audioPath, AVDelayWindow* window)
{
    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device videoDev(videoPath);

    if (!videoDev.isOpen()) {
        std::cerr << "Cannot open device: " << videoDev.errorString() << std::endl;
        return -1;
    }

    if (videoDev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << videoDev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    std::cout << "Openning " << audioPath << std::endl;
    ALSADevice audioDev(audioPath, SND_PCM_STREAM_CAPTURE);

    if (!audioDev.isOpen()) {
        std::cerr << "Cannot open audio device: " << audioDev.errorString() << std::endl;
        return -1;
    }

    v4l2_format format;

    format = setupFormat(videoDev);
    videoDev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << videoDev.buffers().size() << " buffers" << std::endl;

    setupAudio(audioDev);
    preroll(videoDev, videoDev.buffers(), format.fmt.pix_mp);
    mainLoop(videoDev, audioDev, format.fmt.pix_mp, window);

    videoDev.freeBuffers();

    return 0;
}

int main(int argc, char** argv)
{
    std::string videoPath, audioPath;
    if (argc != 3) {
        std::cout << "Usage: " << argv[0] << " video_device audio_device" << std::endl;
        return 0;
    }

    videoPath = argv[1];
    audioPath = argv[2];
    QApplication app(argc, argv);
    AVDelayWindow window;
    std::thread worker(videoThread, videoPath, audioPath, &window);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    window.show();
    int result = app.exec();
    worker.join();

    return result;
}

#include "av-delay-viewer.moc"
