#include "Board.hpp"
#include "FD2110Helper.hpp"
#include "RTPParser.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <arpa/inet.h>
#include <chrono>
#include <csignal>
#include <cstring>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <net/if.h>
#include <poll.h>
#include <sys/ioctl.h>

/*!
 * \dir ./
 * This example shows complex video+audio capture example for FD2110
 */

namespace {
static const int BUFFER_COUNT = 8;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

v4l2_forward_complex_audio_format audioType(V4L2Device& dev)
{
    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_COMPLEX_AUDIO_FORMAT;
    ctl.value = V4L_FORWARD_COMPLEX_AUDIO_ST299;

    if (dev.ioctl(VIDIOC_G_CTRL, &ctl))
        std::cerr << "Cannot get audio format: " << dev.errorString() << std::endl;

    return (v4l2_forward_complex_audio_format)ctl.value;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Each field are in separate buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_COMPLEX_8BIT;
    // 10-bit
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_COMPLEX_10BIT;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt
              << " plane sizes: " << fmt.fmt.pix_mp.plane_fmt[0].sizeimage << " "
              << fmt.fmt.pix_mp.plane_fmt[1].sizeimage << " "
              << fmt.fmt.pix_mp.plane_fmt[2].sizeimage << std::endl;

    return fmt;
}

int setupVideoStream(FD2110Helper& fd2110, V4L2Device& dev, const std::string& videoAddr,
    uint16_t videoPort, const std::string& audioAddr, uint16_t audioPort)
{
    int result;

    result = fd2110.setVideoAddress(dev, videoAddr, videoPort);
    if (result) {
        std::cerr << "Cannot set video stream address: " << dev.errorString() << std::endl;
        return dev.error();
    }

    result = fd2110.setComplexAudioAddress(dev, audioAddr, audioPort);
    if (result) {
        std::cerr << "Cannot set audio stream address: " << dev.errorString() << std::endl;
        return dev.error();
    }

    struct v4l2_control ctl;
    memset(&ctl, 0, sizeof(ctl));

    // Setup input stream color mode + bit depth - should be same as in SDP.
    // Board can do some conversion between stream and v4l2 device format (e.g. 10->8 bit
    //   conversion).
    // Set YCbCr 4:2:2, 10-bit - default format for SDI->2110 converters like AJA.
    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_COLOR;
    ctl.value = V4L_FORWARD_FD2110_STREAM_COLOR_YCBCR422;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream color format: " << dev.errorString() << std::endl;
        return dev.error();
    }

    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_BPC;
    ctl.value = V4L_FORWARD_FD2110_STREAM_BPC_10BIT;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream bpc: " << dev.errorString() << std::endl;
        return dev.error();
    }

    return 0;
}

int setupStream(FD2110Helper& fd2110, int stream, const std::string& videoAddr, uint16_t videoPort,
    const std::string& audioAddr, uint16_t audioPort, std::string& videoDev, int& socketfd)
{
    int result = fd2110.enableStream(stream);
    if (result) {
        std::cerr << "Cannot enable stream " << stream << std::endl;
        return result;
    }
    videoDev = fd2110.getVideoDevice(stream);
    std::cout << "Stream #" << stream << " enabled, video device: " << videoDev << std::endl;

    // Setup V4L2 device - pass stream address and format to driver
    V4L2Device video(videoDev);
    if (!video.isOpen()) {
        std::cerr << "Cannot open video device: " << video.errorString() << std::endl;
        return -1;
    }
    std::cout << "Configuring V4L2 device..." << std::endl;
    result = setupVideoStream(fd2110, video, videoAddr, videoPort, audioAddr, audioPort);
    if (result)
        return result;

    // Setup network - open socket for IP filter setup and multicast join
    result = fd2110.joinMulticast(stream, videoAddr, videoPort);
    if (result)
        return result;

    return 0;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill playback queue
    for (auto buf : buffers)
        dev.queueBuffer(buf);
}

void mainLoop(V4L2Device& dev)
{
    int type = dev.type();

    // Start capture
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLIN;

    RTPParser parser;
    v4l2_forward_complex_audio_format atype = audioType(dev);

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLIN))
            continue;

        // Get buffer from queue
        V4L2Device::BufferPtr b = dev.dequeueBuffer();

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Received buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << b->v4l2buf().index << ", sequence = " << b->v4l2buf().sequence
                  << ", field = " << b->v4l2buf().field << ", timestamp = "
                  << (uint64_t)b->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + b->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, video size = " << b->v4l2buf().m.planes[0].bytesused
                  << ", anc size = " << b->v4l2buf().m.planes[1].bytesused
                  << ", meta size = " << b->v4l2buf().m.planes[2].bytesused << std::endl;

        /*
         * PROCESS BUFFER DATA HERE
         */
        auto start = std::chrono::high_resolution_clock::now();
        // Parse all ANC packets
        parser.parse(b->data(1), b->bytesused(1));

        // Extract Audio
        uint32_t audio[4 * 2048];
        int samples = parser.extractST2110AudioPCM24(audio, 4, 2048, 97);
        auto dt = std::chrono::high_resolution_clock::now() - start;

        std::cout << "RTPParser done in "
                  << std::chrono::duration_cast<std::chrono::nanoseconds>(dt).count()
                  << "ns Audio: " << samples << " samples" << std::endl;

        const struct forward_v4l2_complex_info* info
            = (struct forward_v4l2_complex_info*)b->data(2);
        std::cout << "Complex info seq = " << info->seq << ", field = " << info->field
                  << ", timestamp = " << info->timestamp
                  << ", hw_timestamp = " << info->hw_timestamp
                  << ", video size = " << info->video_size << ", vbi size = " << info->vbi_size
                  << ", anc size = " << info->anc_size << std::endl;

        // Return buffer back to queue
        dev.queueBuffer(b);
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int main(int argc, char** argv)
{
    std::string board, videoAddr, audioAddr;
    int stream;
    int ethernet;
    uint8_t audioPT;
    uint16_t videoPort, audioPort;
    std::string videoPath;
    int socketfd;

    if (argc != 8) {
        std::cout << "Usage: " << argv[0]
                  << " board stream_idx video_addr video_port audio_addr audio_port audio_pt"
                  << std::endl;
        return 0;
    }

    // FD2110 has (16 input streams + 16 output streams) x 2 Ethernets = 64 streams total
    // Streams represented as I/O pin, in order: 2x SDI, 16x eth0 inputs, 16x eth0 outputs, 16x eth1
    // inputs, 16x eth1 outputs
    board = argv[1];
    stream = std::stoi(argv[2]);
    videoAddr = argv[3];
    videoPort = std::stoul(argv[4]);
    audioAddr = argv[5];
    audioPort = std::stoul(argv[6]);
    audioPT = std::stoi(argv[7]);

    FD2110Helper helper(board);

    // Setup stream parameters - addresses, ports, payload type, pixel format, audio format, etc.
    int result = setupStream(
        helper, stream, videoAddr, videoPort, audioAddr, audioPort, videoPath, socketfd);

    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device dev(videoPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers());
    mainLoop(dev);

    dev.freeBuffers();

    return 0;
}
