#include "ANCParser.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <chrono>
#include <csignal>
#include <cstring>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <poll.h>

/*!
 * \dir ./
 * This example shows complex video+audio+vbi capture example
 */

namespace {
static const int BUFFER_COUNT = 8;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

// VBI should be explicit enabled by controls
void toggleVBI(V4L2Device& dev, bool on)
{
    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_ENABLE_VANC;
    ctl.value = on;

    if (dev.ioctl(VIDIOC_S_CTRL, &ctl))
        std::cerr << "Cannot toggle VANC: " << dev.errorString() << std::endl;
}

// VBI should be explicit enabled by controls
v4l2_forward_complex_audio_format audioType(V4L2Device& dev)
{
    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_COMPLEX_AUDIO_FORMAT;
    ctl.value = V4L_FORWARD_COMPLEX_AUDIO_ST299;

    if (dev.ioctl(VIDIOC_G_CTRL, &ctl))
        std::cerr << "Cannot get audio format: " << dev.errorString() << std::endl;

    return (v4l2_forward_complex_audio_format)ctl.value;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Each field are in separate buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_COMPLEX_8BIT;
    // 10-bit
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_COMPLEX_10BIT;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt
              << " plane sizes: " << fmt.fmt.pix_mp.plane_fmt[0].sizeimage << " "
              << fmt.fmt.pix_mp.plane_fmt[1].sizeimage << " "
              << fmt.fmt.pix_mp.plane_fmt[2].sizeimage << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill playback queue
    for (auto buf : buffers)
        dev.queueBuffer(buf);
}

void mainLoop(V4L2Device& dev)
{
    int type = dev.type();

    // Start capture
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLIN;

    ANCParser parser;
    v4l2_forward_complex_audio_format atype = audioType(dev);

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLIN))
            continue;

        // Get buffer from queue
        V4L2Device::BufferPtr b = dev.dequeueBuffer();

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Received buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << b->v4l2buf().index << ", sequence = " << b->v4l2buf().sequence
                  << ", field = " << b->v4l2buf().field << ", timestamp = "
                  << (uint64_t)b->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + b->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, video size = " << b->v4l2buf().m.planes[0].bytesused
                  << ", anc size = " << b->v4l2buf().m.planes[1].bytesused
                  << ", meta size = " << b->v4l2buf().m.planes[2].bytesused << std::endl;

        /*
         * PROCESS BUFFER DATA HERE
         */
        auto start = std::chrono::high_resolution_clock::now();

        // Parse all ANC packets
        parser.parse(b->data(1), b->bytesused(1), atype == V4L_FORWARD_COMPLEX_AUDIO_ST272);

        // Find VPID packet
        uint32_t vpid;
        parser.extractData8bit(0, 0x41, (uint8_t*)&vpid, sizeof(vpid));

        // Extract Audio
        uint32_t audio[2 * 2048];
        int samples = 0;
        if (atype == V4L_FORWARD_COMPLEX_AUDIO_ST272)
            samples = parser.extractSDAudioPCM(audio, 2, 2048);
        else
            samples = parser.extractHDAudioPCM(audio, 2, 2048);

        auto dt = std::chrono::high_resolution_clock::now() - start;

        std::cout << "ANCParser done in "
                  << std::chrono::duration_cast<std::chrono::nanoseconds>(dt).count()
                  << "ns VPID: " << std::hex << std::setfill('0') << std::setw(8)
                  << __builtin_bswap32(vpid) << std::dec << " Audio: " << samples << " samples"
                  << std::endl;

        const struct forward_v4l2_complex_info* info
            = (struct forward_v4l2_complex_info*)b->data(2);
        std::cout << "Complex info seq = " << info->seq << ", field = " << info->field
                  << ", timestamp = " << info->timestamp
                  << ", hw_timestamp = " << info->hw_timestamp
                  << ", video size = " << info->video_size << ", vbi size = " << info->vbi_size
                  << ", anc size = " << info->anc_size << ", video stride = " << info->video_stride
                  << ", vbi stride = " << info->vbi_stride << std::endl;

        // Return buffer back to queue
        dev.queueBuffer(b);
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    toggleVBI(dev, true);
    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers());
    mainLoop(dev);

    dev.freeBuffers();

    return 0;
}
