#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"

#include <QApplication>
#include <QImage>
#include <QPainter>
#include <QWidget>

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>
#include <thread>

/*!
 * \dir ./
 * This example shows simple video capture with video preview using Qt5
 */

namespace {
static const int BUFFER_COUNT = 8;
}

static bool shouldStop = false;

// Qt Window
class PreviewWindow : public QWidget {
    Q_OBJECT
public:
    PreviewWindow(QWidget* parent = nullptr)
    {
    }

public:
    void updateImage(const QImage& img)
    {
        m_image = img;
        update();
    }

protected:
    void paintEvent(QPaintEvent* event) override
    {
        QPainter p(this);

        if (m_image.isNull())
            return;

        float a = (float)m_image.width() / (float)m_image.height();
        QRect window = QRect(0, 0, width(), height());
        QRect target;

        if ((window.width() / a) > window.height())
            target = QRect(0, 0, window.height() * a, window.height());
        else
            target = QRect(0, 0, window.width(), window.width() / a);

        target.moveTo(
            (window.width() - target.width()) / 2, (window.height() - target.height()) / 2);

        p.drawImage(target, m_image);
    }
    void closeEvent(QCloseEvent* event) override
    {
        shouldStop = true;
    }

private:
    QImage m_image;
};

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
    qApp->quit();
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.ppix_mpx.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers,
    const v4l2_pix_format_mplane& fmt)
{
    // Fill playback queue
    for (auto buf : buffers) {
        dev.queueBuffer(buf);
    }
}

void mainLoop(V4L2Device& dev, const v4l2_pix_format_mplane& fmt, PreviewWindow* window)
{
    int type = dev.type();

    QImage image = QImage(fmt.width, fmt.height, QImage::Format_RGB888);
    image.fill(Qt::red);

    // Start capture
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLIN;

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLIN))
            continue;

        // Get buffer from queue
        auto buf = dev.dequeueBuffer();

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Received buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << buf->v4l2buf().index
                  << ", sequence = " << buf->v4l2buf().sequence
                  << ", field = " << buf->v4l2buf().field << ", timestamp = "
                  << (uint64_t)buf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + buf->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, size = " << buf->bytesused(0) << std::endl;

        // Convert to RGB
        convertVideoToRGB888(fmt, buf->data(0), image.bits());

        // Pass it to window
        QMetaObject::invokeMethod(
            window, [=] { window->updateImage(image); }, Qt::QueuedConnection);

        // Return buffer back to queue
        dev.queueBuffer(buf);
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int videoThread(std::string devPath, PreviewWindow* window)
{
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    v4l2_format format;

    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers(), format.fmt.pix_mp);
    mainLoop(dev, format.fmt.pix_mp, window);

    dev.freeBuffers();

    return 0;
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    QApplication app(argc, argv);
    PreviewWindow window;
    std::thread worker(videoThread, devPath, &window);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    window.show();
    int result = app.exec();
    worker.join();

    return result;
}

#include "capture-preview.moc"
