#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"

#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <poll.h>

/*!
 * \dir ./
 * This example shows simple video capture example
 */

namespace {
static const int BUFFER_COUNT = 8;
}

static bool shouldStop = false;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Capture frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill playback queue
    for (auto buf : buffers)
        dev.queueBuffer(buf);
}

void mainLoop(V4L2Device& dev)
{
    int type = dev.type();

    // Start capture
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLIN;

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLIN))
            continue;

        // Get buffer from queue
        V4L2Device::BufferPtr b = dev.dequeueBuffer();

        /*
         * PROCESS BUFFER DATA HERE
         */

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Received buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << b->v4l2buf().index << ", sequence = " << b->v4l2buf().sequence
                  << ", field = " << b->v4l2buf().field << ", timestamp = "
                  << (uint64_t)b->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + b->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, size = " << b->v4l2buf().m.planes[0].bytesused << std::endl;

        // Return buffer back to queue
        dev.queueBuffer(b);
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an input device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    std::cout << "Capturing with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers());
    mainLoop(dev);

    dev.freeBuffers();

    return 0;
}
