#include "Board.hpp"

#include <algorithm>
#include <iostream>
#include <libudev.h>
#include <map>
#include <regex>
#include <tuple>

#include "../../../include/forward-v4l2-ioctl.h"
#include <linux/videodev2.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>

using UdevDevice = std::shared_ptr<struct udev_device>;
using Info = Board::Info;
using IOMode = Board::IOMode;

namespace {
static const char* FORWARD_CLASS = "forward";
static const char* FORWARD_TYPE_ATTR = "dev_type";
static const char* FORWARD_IO_ATTR = "io_config";
static const std::regex ALSA_PCM_REGEXP("pcmC(\\d+)D(\\d+)[cp]");

static std::map<uint16_t, std::tuple<Info::Type, int>> FORWARD_DEVICE_IDS {
    { 0x0019, { Board::Info::FD722, 4 } },
    { 0x001a, { Board::Info::FD788, 8 } },
    { 0x001b, { Board::Info::FD720, 2 } },
    { 0x001c, { Board::Info::FD922, 4 } },
    { 0x001f, { Board::Info::FD940, 4 } },
    { 0x0020, { Board::Info::FD2110, 66 } },
    { 0x0021, { Board::Info::FD722, 4 } },
    { 0x0022, { Board::Info::FD788, 8 } },
    { 0x0023, { Board::Info::FD722M2, 3 } },
    { 0x0024, { Board::Info::FD722BP, 4 } },
    { 0x0025, { Board::Info::FD922, 4 } },
    { 0x0026, { Board::Info::FD788, 8 } },
    { 0x0027, { Board::Info::FD940, 4 } },
};
}

static struct std::shared_ptr<struct udev> udev
    = std::shared_ptr<struct udev>(udev_new(), &udev_unref);
static Info infoFromUdev(UdevDevice device);

static std::string getAttribute(UdevDevice device, const std::string& attr)
{
    const char* result = udev_device_get_sysattr_value(device.get(), attr.c_str());

    if (!result)
        return std::string();
    else
        return std::string(result);
}

static int setAttribute(UdevDevice device, const std::string& attr, const std::string& value)
{
    return udev_device_set_sysattr_value(device.get(), attr.c_str(), value.c_str());
}

/*! \class Board
    This class represents a board with several inputs/outputs (I/Os).

    Main purpose of this class is to configure board-specific parameters.
    Notable example is switching I/O direction - some boards (FD788) has pins that can being either
   input or outputs. Each board has several I/Os, ordered by physical location - top->bottom or
   right->left, inputs first.

    Main information about board is obtainable through \ref Info structure.
    This structure contains common info and list of V4L2 and ALSA devices available on board.
    Video/VBI/ALSA device has same ordering as physical pins - e.g. IN1 = /dev/video0 + /dev/vbi0 +
   hw:0,1

    Enumeration of all boards in system can be done using static \ref enumerate()
*/
Board::Board(const Info& board) : m_info(board)
{
}
/*!
 * \return board info
 */
const Info& Board::info() const
{
    return m_info;
}

/*!
 * \return number of I/O pins, available on board
 */
int Board::ioCount() const
{
    return m_info.ioCount;
}

/*!
 * Return modes of all I/Os, ordered by physical location - [0] = top-most (right-most) pin
 * \return list of I/O modes, ordered by physical location
 */
std::vector<IOMode> Board::ioMode() const
{
    std::vector<IOMode> modes;
    modes.resize(ioCount(), IOMode::Disabled);

    if (!m_info.udev)
        return modes;

    std::string cfg = getAttribute(m_info.udev, FORWARD_IO_ATTR);
    for (int i = 0; (i < (int)modes.size()) && (i < (int)cfg.length()); i++) {
        if ((cfg[i] == 'I') || (cfg[i] == 'i'))
            modes[i] = IOMode::Input;
        else if ((cfg[i] == 'O') || (cfg[i] == 'o'))
            modes[i] = IOMode::Output;
    }

    return modes;
}

/*!
 * Switch I/O mode - I/O can be either Input or Output.
 * Input <-> Output switching is available only on boards with bidirectional I/O (only FD788 for
 * now) \arg io - I/O pin number, ordered by physical location, [0] = top-most (right-most) pin \arg
 * mode - I/O pin mode \return 0 if switching is successful, errno otherwise
 */
int Board::switchIOMode(int io, IOMode mode)
{
    auto modes = ioMode();
    if (io >= (int)modes.size())
        return -EINVAL;
    modes[io] = mode;
    return configureIOMode(modes);
}

/*!
 * Switch I/O mode for all pins
 * Same as \ref switchIOMode(), but for all pins
 * \arg modes - I/O pin modes
 * \return 0 if switching is successful, errno otherwise
 */
int Board::configureIOMode(std::vector<IOMode> modes)
{
    std::string cfg = getAttribute(m_info.udev, FORWARD_IO_ATTR);

    for (int i = 0; (i < (int)modes.size()) && (i < (int)cfg.length()); i++) {
        if (modes[i] == IOMode::Input)
            cfg[i] = 'I';
        else if (modes[i] == IOMode::Output)
            cfg[i] = 'O';
        else
            cfg[i] = '-';
    }
    int result = setAttribute(m_info.udev, FORWARD_IO_ATTR, cfg);
    m_info = infoFromUdev(m_info.udev);
    return result;
}

static std::vector<UdevDevice> getBoardChildren(UdevDevice device)
{
    std::vector<UdevDevice> result;
    struct udev_enumerate* cenum = udev_enumerate_new(udev.get());

    UdevDevice root = UdevDevice(udev_device_get_parent(device.get()), [](void* p) { });
    std::string devSysPath = udev_device_get_syspath(device.get());
    std::string rootSysPath = udev_device_get_syspath(root.get());

    udev_enumerate_add_match_parent(cenum, root.get());
    udev_enumerate_scan_devices(cenum);

    struct udev_list_entry *cdevs, *cdev_it;
    cdevs = udev_enumerate_get_list_entry(cenum);

    udev_list_entry_foreach(cdev_it, cdevs)
    {
        std::string syspath = udev_list_entry_get_name(cdev_it);

        if ((syspath == devSysPath) || (syspath == rootSysPath))
            continue;

        UdevDevice cdev = UdevDevice(
            udev_device_new_from_syspath(udev.get(), syspath.c_str()), &udev_device_unref);
        result.push_back(cdev);
    }

    udev_enumerate_unref(cenum);

    return result;
}

static Info infoFromUdev(UdevDevice device)
{
    Info info;
    struct udev_device* pciDevice;

    if (!device)
        return info;

    pciDevice = udev_device_get_parent(device.get());

    info.idDevice = std::stoul(udev_device_get_sysattr_value(pciDevice, "device"), nullptr, 0);
    info.idVendor = std::stoul(udev_device_get_sysattr_value(pciDevice, "vendor"), nullptr, 0);

    auto it = FORWARD_DEVICE_IDS.find(info.idDevice);

    if (it != FORWARD_DEVICE_IDS.end()) {
        info.type = std::get<0>(it->second);
        info.ioCount = std::get<1>(it->second);
    } else {
        info.type = Info::Unknown;
        info.ioCount = 0;
    }
    info.videoDevs = std::vector<std::string>(info.ioCount);
    info.vbiDevs = std::vector<std::string>(info.ioCount);
    info.audioDevs = std::vector<std::string>(info.ioCount);
    info.netDevs = std::vector<std::string>(info.ioCount);

    info.name = getAttribute(device, "name");
    info.softwareID = std::stoll(getAttribute(device, "software_id"));
    info.hardwareID = std::stoull(getAttribute(device, "hardware_id"), nullptr, 16);
    info.path = udev_device_get_devnode(device.get());
    info.udev = device;

    auto children = getBoardChildren(device);
    std::vector<std::tuple<std::string, std::string, std::string, int>> devNames;

    for (auto child : children) {
        std::string subsystem, devnode, sysname;

        const char* str = udev_device_get_subsystem(child.get());
        if (!str)
            continue;
        subsystem = str;

        str = udev_device_get_devnode(child.get());
        if (str)
            devnode = str;
        str = udev_device_get_sysname(child.get());
        if (str)
            sysname = str;

        if (subsystem == "video4linux") {
            bool vbi = false;
            int io = 0;
            int vfd = open(devnode.c_str(), O_RDONLY);
            if (vfd > 0) {
                struct v4l2_capability cap = { 0 };
                ioctl(vfd, VIDIOC_QUERYCAP, &cap);
                vbi = (cap.device_caps & (V4L2_CAP_VBI_CAPTURE | V4L2_CAP_VBI_OUTPUT)) ? true
                                                                                       : false;
                ioctl(vfd, VIDIOC_FORWARD_GET_IO_INDEX, &io);
                close(vfd);
            }

            if (vbi)
                info.vbiDevs[io] = devnode;
            else
                info.videoDevs[io] = devnode;
        } else if (subsystem == "sound") {
            std::smatch rm;
            if (!std::regex_match(sysname, rm, ALSA_PCM_REGEXP))
                continue;

            int io = std::atoi(rm[2].str().c_str());

            info.audioDevs[io] = std::string("hw:") + rm[1].str() + "," + rm[2].str();
        } else if (subsystem == "net") {
            int port = 0;
            str = udev_device_get_sysattr_value(child.get(), "dev_port");
            if (str)
                port = std::atoi(str);
            if (info.type == Info::FD2110) {
                for (int i = 0; i < 32; i++)
                    info.netDevs[(port - 1) * 32 + i + 2] = sysname;
            }
        }
    }

    return info;
}

static Info getInfo(const std::string& syspath)
{
    UdevDevice device;
    Info info;

    device
        = UdevDevice(udev_device_new_from_syspath(udev.get(), syspath.c_str()), &udev_device_unref);

    if (!device)
        return info;

    info = infoFromUdev(device);

    return info;
}

static void processEnumerate(struct udev_enumerate* enumerate, std::vector<Info>& devs)
{
    struct udev_list_entry *devices, *device_it;

    devices = udev_enumerate_get_list_entry(enumerate);

    udev_list_entry_foreach(device_it, devices)
    {
        UdevDevice dev = UdevDevice(
            udev_device_new_from_syspath(udev.get(), udev_list_entry_get_name(device_it)),
            &udev_device_unref);

        Info info;

        const char* path = udev_list_entry_get_name(device_it);
        info = getInfo(path);

        devs.push_back(info);
    }
}

std::vector<Board::Info> Board::enumerate()
{
    std::vector<Info> devs;

    struct udev_enumerate* enumerate;

    enumerate = udev_enumerate_new(udev.get());
    udev_enumerate_add_match_subsystem(enumerate, FORWARD_CLASS);
    udev_enumerate_scan_devices(enumerate);
    processEnumerate(enumerate, devs);
    udev_enumerate_unref(enumerate);

    return devs;
}

Info Board::fromPath(const std::string& path)
{
    struct udev_enumerate* enumerate;
    Info info;
    info.type = Info::Unknown;

    enumerate = udev_enumerate_new(udev.get());
    udev_enumerate_add_match_subsystem(enumerate, FORWARD_CLASS);
    udev_enumerate_scan_devices(enumerate);
    struct udev_list_entry *devices, *device_it;

    devices = udev_enumerate_get_list_entry(enumerate);
    udev_list_entry_foreach(device_it, devices)
    {
        UdevDevice dev = UdevDevice(
            udev_device_new_from_syspath(udev.get(), udev_list_entry_get_name(device_it)),
            &udev_device_unref);

        const char* p = udev_device_get_devnode(dev.get());

        if (!p || (path != p))
            continue;

        info = infoFromUdev(dev);
        break;
    }

    udev_enumerate_unref(enumerate);

    return info;
}
