#pragma once

#include <cstdint>
#include <memory>
#include <string>
#include <vector>

extern "C" {
struct udev_device;
}

class Board {
public:
    struct Info {
        //! Board type
        enum Type { Unknown, FD722, FD788, FD720, FD922, FD940, FD2110, FD722M2, FD722BP };
        //! Board type
        Type type;
        //! Board device path (/dev/forward/*)
        std::string path;
        //! PCI vendor id (0x1c1f typical)
        uint16_t idVendor;
        //! PCI device id
        uint16_t idDevice;
        //! Board full name (e.g. "FD788 80002")
        std::string name;
        //! Board software id, same as on board label
        int64_t softwareID;
        //! Board hardware serial id
        uint64_t hardwareID;
        //! Video I/O count
        int ioCount;
        //! V4L2 video device nodes ("/dev/video*"), ordered by I/O number, inputs first
        std::vector<std::string> videoDevs;
        //! V4L2 VBI device nodes ("/dev/vbi*"), in same order as \ref videoDevs
        std::vector<std::string> vbiDevs;
        //! ALSA device nodes names ("hw:X,Y"), in same order as \ref videoDevs
        std::vector<std::string> audioDevs;
        //! Net device names ("ethX"), in same order as \ref videoDevs
        std::vector<std::string> netDevs;
        //! Board udev node, used by \ref Board class
        std::shared_ptr<struct udev_device> udev;
    };

    //! I/O mode
    enum class IOMode { Disabled, Input, Output };

    //! Create board from info, provided by \ref enumerate
    Board(const Info& board);

    //! Board info
    const Info& info() const;

    //! Board I/O count
    int ioCount() const;
    //! Get current I/O mode
    std::vector<IOMode> ioMode() const;
    //! Switch I/O to specific mode
    int switchIOMode(int io, IOMode mode);
    //! Configure all I/O
    int configureIOMode(std::vector<IOMode> modes);

    //! Enumerate all boards in system
    static std::vector<Info> enumerate();
    //! Get board info from /dev path
    static Info fromPath(const std::string& path);

private:
    Info m_info;
};
