#include "FD2110Helper.hpp"
#include "forward-v4l2-ioctl.h"

#include <arpa/inet.h>
#include <cstring>
#include <iostream>
#include <net/if.h>
#include <sys/ioctl.h>
#include <unistd.h>

FD2110Helper::FD2110Helper(const std::string& board)
{
    Board::Info info = Board::fromPath(board);
    int result = 0;

    if (info.type == Board::Info::Unknown) {
        std::cerr << "Wrong board path, should be /dev/forward/fd2110-*" << std::endl;
        return;
    } else if (info.type != Board::Info::FD2110) {
        std::cerr << "Not a FD2110 board" << std::endl;
        return;
    }

    if (info.type == Board::Info::FD2110)
        m_board = std::make_unique<Board>(info);
}

FD2110Helper::~FD2110Helper()
{
    if (m_sock != -1)
        close(m_sock);
}

int FD2110Helper::enableStream(int stream)
{
    if (!m_board)
        return -ENODEV;

    return m_board->switchIOMode(
        stream + 2, ((stream % 32) / 16) ? Board::IOMode::Output : Board::IOMode::Input);
}

std::string FD2110Helper::getVideoDevice(int stream)
{
    if (!m_board)
        return std::string();

    Board::Info info = m_board->info();
    int i = stream + 2;

    if (i >= info.videoDevs.size())
        return std::string();

    return info.videoDevs[i];
}

std::string FD2110Helper::getAudioDevice(int stream)
{
    if (!m_board)
        return std::string();

    Board::Info info = m_board->info();
    int i = stream + 2;

    if (i >= info.audioDevs.size())
        return std::string();

    return info.audioDevs[i];
}

static bool str2ipv6(const std::string& str, uint8_t* ip, bool& v6)
{
    struct in_addr addr;
    struct in6_addr addr6;

    memset(ip, 0, 16);

    if (inet_pton(AF_INET, str.c_str(), &addr)) {
        uint8_t* in_p = (uint8_t*)&addr.s_addr;

        for (int i = 0; i < 4; i++)
            ip[12 + i] = in_p[i];
        ip[11] = 0xFF;
        ip[10] = 0xFF;
        v6 = false;
    } else if (inet_pton(AF_INET6, str.c_str(), &addr6)) {
        for (int i = 0; i < 16; i++)
            ip[i] = addr6.s6_addr[i];
        v6 = true;
    } else
        return false;

    return true;
}

int FD2110Helper::setVideoAddress(V4L2Device& dev, const std::string& videoAddr, uint16_t videoPort)
{
    struct v4l2_forward_stream_address v4l2addr;
    bool v6;

    if (!str2ipv6(videoAddr, v4l2addr.ipv6, v6))
        return -EINVAL;
    v4l2addr.port = videoPort;

    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;

    // Setup IP + Port
    memset(&ex_ctl, 0, sizeof(ex_ctl));
    memset(&ex_ctls, 0, sizeof(ex_ctls));

    ex_ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_ADDRESS;
    ex_ctl.size = sizeof(v4l2addr);
    ex_ctl.ptr = &v4l2addr;
    ex_ctls.ctrl_class = V4L2_CTRL_ID2CLASS(V4L2_CID_FORWARD_FD2110_STREAM_ADDRESS);
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    if (dev.ioctl(VIDIOC_S_EXT_CTRLS, &ex_ctls) < 0)
        return dev.error();

    return 0;
}

int FD2110Helper::setComplexAudioAddress(
    V4L2Device& dev, const std::string& audioAddress, uint16_t audioPort)
{
    struct v4l2_forward_stream_address v4l2addr;
    bool v6;

    if (!str2ipv6(audioAddress, v4l2addr.ipv6, v6))
        return -EINVAL;
    v4l2addr.port = audioPort;

    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;
    struct v4l2_control ctl;

    // Setup IP + Port
    memset(&ex_ctl, 0, sizeof(ex_ctl));
    memset(&ex_ctls, 0, sizeof(ex_ctls));
    memset(&ctl, 0, sizeof(ctl));

    ex_ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_AUDIO_ADDRESS;
    ex_ctl.size = sizeof(v4l2addr);
    ex_ctl.ptr = &v4l2addr;
    ex_ctls.ctrl_class = V4L2_CTRL_ID2CLASS(V4L2_CID_FORWARD_FD2110_STREAM_ADDRESS);
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    if (dev.ioctl(VIDIOC_S_EXT_CTRLS, &ex_ctls) < 0)
        return dev.error();

    return 0;
}

int FD2110Helper::joinMulticast(int stream, const std::string& addr, uint16_t port)
{
    int optval;
    struct sockaddr_in6 saddr;
    int result;
    bool v6;
    int i = stream + 2;

    std::string iface = m_board->info().netDevs[i];

    // Set socket address
    memset(&saddr, 0, sizeof(saddr));
    if (!str2ipv6(addr, saddr.sin6_addr.s6_addr, v6))
        return -EINVAL;

    saddr.sin6_family = AF_INET6;
    saddr.sin6_port = htons(port);

    // Open socket
    m_sock = socket(AF_INET6, SOCK_DGRAM, 0);
    if (m_sock < 0) {
        return -errno;
    }

    // Allow both IPv6 and IPv4 on same socket
    optval = 0;
    result = setsockopt(m_sock, IPPROTO_IPV6, IPV6_V6ONLY, &optval, sizeof(optval));
    if (result)
        return result;

    // Bind socket to device
    result = setsockopt(m_sock, SOL_SOCKET, SO_BINDTODEVICE, iface.c_str(), iface.length());
    if (result)
        return result;

    // Bind socket to address
    result = bind(m_sock, (sockaddr*)&saddr, sizeof(saddr));
    if (result) {
        std::cerr << "Socket binding failed: " << strerror(errno) << std::endl;
        return result;
    }

    // Find net interface index by name
    struct ifreq ifidx;
    memset(&ifidx, 0, sizeof(ifidx));
    strncpy(ifidx.ifr_name, iface.c_str(), IFNAMSIZ - 1);
    result = ioctl(m_sock, SIOCGIFINDEX, &ifidx);
    if (result < 0) {
        std::cerr << "Cannot find net interface index" << std::endl;
        return -EINVAL;
    }

    // Allow address reuse
    optval = 1;
    result = setsockopt(m_sock, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));
    if (result) {
        std::cerr << "SO_REUSEADDR failed: " << strerror(errno) << std::endl;
        return result;
    }

    // Join multicast
    if (v6) {
        struct ipv6_mreq mreq;
        mreq.ipv6mr_multiaddr = saddr.sin6_addr;
        mreq.ipv6mr_interface = ifidx.ifr_ifindex;
        result = setsockopt(m_sock, IPPROTO_IPV6, IPV6_ADD_MEMBERSHIP, &mreq, sizeof(mreq));
        if (result) {
            std::cerr << "IPV6_ADD_MEMBERSHIP failed: " << strerror(errno) << std::endl;
            return result;
        }
    } else {
        struct ip_mreqn mreq;
        mreq.imr_multiaddr.s_addr = *((in_addr_t*)&saddr.sin6_addr.s6_addr[12]);
        mreq.imr_address.s_addr = INADDR_ANY;
        mreq.imr_ifindex = ifidx.ifr_ifindex;
        result = setsockopt(m_sock, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq));
        if (result) {
            std::cerr << "IP_ADD_MEMBERSHIP failed: " << strerror(errno) << std::endl;
            return result;
        }
    }

    return 0;
}
