#include "Utils.hpp"

#include "forward-v4l2-ioctl.h"
#include <iomanip>

std::ostream& operator<<(std::ostream& str, const v4l2_dv_timings& tim)
{
    int width = tim.bt.width;
    int height = tim.bt.height;
    int fps100 = (uint64_t)tim.bt.pixelclock * 100 / V4L2_DV_BT_FRAME_WIDTH(&tim.bt)
        / V4L2_DV_BT_FRAME_HEIGHT(&tim.bt);
    bool interlaced = tim.bt.interlaced ? true : false;
    if (interlaced)
        fps100 *= 2;

    // Modes with /1.001 framerate are designated by V4L2_DV_FL_REDUCED_FPS flag
    if (tim.bt.flags & V4L2_DV_FL_REDUCED_FPS)
        fps100 = fps100 * 1000 / 1001;

    str << width << "x" << height << (interlaced ? "i" : "p") << fps100 / 100;
    if ((fps100 % 100) != 0)
        str << "." << std::setw(2) << std::setfill('0') << (fps100 % 100);

    return str;
}

std::ostream& operator<<(std::ostream& str, const v4l2_format& fmt)
{
    if ((fmt.type == V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE)
        || (fmt.type == V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE)) {
        v4l2_pix_format_mplane pix = fmt.fmt.pix_mp;
        char fourcc[5] = { '0', '0', '0', '0' };
        for (int i = 0; i < 4; i++)
            fourcc[i] = (pix.pixelformat >> (i * 8)) & 0xFF;

        str << "(" << pix.width << "x" << pix.height;
        switch (pix.field) {
        case V4L2_FIELD_INTERLACED:
            str << ", interlaced";
            break;
        case V4L2_FIELD_ALTERNATE:
            str << ", alternate";
            break;
        case V4L2_FIELD_NONE:
            str << ", progressive";
            break;
        }

        str << ", " << fourcc << ", " << pix.plane_fmt[0].sizeimage << ")";
    } else if ((fmt.type == V4L2_BUF_TYPE_VBI_CAPTURE) || (fmt.type == V4L2_BUF_TYPE_VBI_OUTPUT)) {
        v4l2_vbi_format vbi = fmt.fmt.vbi;
        char fourcc[5] = { '0', '0', '0', '0' };
        for (int i = 0; i < 4; i++)
            fourcc[i] = (vbi.sample_format >> (i * 8)) & 0xFF;

        str << "(" << vbi.sampling_rate << "Hz, " << vbi.start[0] << "+" << vbi.count[0] << ", "
            << vbi.start[1] << "+" << vbi.count[1] << ", " << vbi.samples_per_line << "spl"
            << ")";
    }
    return str;
}

void generateRP219ColorBarUYVY(uint8_t* buf, uint32_t pixelformat, int width, int height)
{
    int a = width, b = height;
    double c = a * 3.0 / 4.0 / 7.0;
    double d = a / 4.0 / 2.0;

    if (pixelformat == V4L2_PIX_FMT_UYVY) {
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x += 2) {
                uint32_t* out = (uint32_t*)&buf[(y * width + x) * 2];
                if (x < d) {
                    *out = 0x68806880;
                } else if (x < d + 1 * c) {
                    *out = 0xB480B480;
                } else if (x < d + 2 * c) {
                    *out = 0xA888A82C;
                } else if (x < d + 3 * c) {
                    *out = 0x912C9193;
                } else if (x < d + 4 * c) {
                    *out = 0x8534853F;
                } else if (x < d + 5 * c) {
                    *out = 0x3FCC3FC1;
                } else if (x < d + 6 * c) {
                    *out = 0x33D4336D;
                } else if (x < d + 7 * c) {
                    *out = 0x1D781DD4;
                } else {
                    *out = 0x68806880;
                }
            }
        }
    } else if (pixelformat == V4L2_PIX_FMT_V210) {
        for (int y = 0; y < height; y++) {
            for (int x = 0; x < width; x += 3) {
                uint64_t* out = (uint64_t*)&buf[(y * width + x) * 8 / 3];
                bool cbcr = (x / 3) % 2;
                if (x < d) {
                    *out = 0x1A0801A020068200ULL;
                } else if (x < d + 1 * c) {
                    *out = 0x2D0802D0200B4200ULL;
                } else if (x < d + 2 * c) {
                    *out = cbcr ? 0x2A0882A00B0A8220ULL : 0x2A02C2A0220A80B0ULL;
                } else if (x < d + 3 * c) {
                    *out = cbcr ? 0x2442C24424C910B0ULL : 0x244932440B09124CULL;
                } else if (x < d + 4 * c) {
                    *out = cbcr ? 0x214342140FC850D0ULL : 0x2143F2140D0850FCULL;
                } else if (x < d + 5 * c) {
                    *out = cbcr ? 0x0FCCC0FC3043F330ULL : 0x0FCC10FC3303F304ULL;
                } else if (x < d + 6 * c) {
                    *out = cbcr ? 0x0CCD40CC1B433350ULL : 0x0CC6D0CC350331B4ULL;
                } else if (x < d + 7 * c) {
                    *out = cbcr ? 0x074780743501D1E0ULL : 0x074D40741E01D350ULL;
                } else {
                    *out = 0x1A0801A020068200ULL;
                }
            }
        }
    }
}

static inline uint8_t _clamp8bit_full(float v)
{
    return (v < 0.0f) ? 0 : ((v > 255.0f) ? 255 : v);
}

static inline void _ycbcr888_rgb888_bt709(
    uint8_t y, uint8_t cb, uint8_t cr, uint8_t& r, uint8_t& g, uint8_t& b)
{
    float yf = ((float)y - 16) * 1.1644f;
    float cbf = ((float)cb - 128.0f) * 1.1384f;
    float crf = ((float)cr - 128.0f) * 1.1384f;
    r = _clamp8bit_full(yf + 1.5748f * crf);
    g = _clamp8bit_full(
        yf - (0.2126f * 1.5748f / 0.7152f) * crf - (0.0722f * 1.8556f / 0.7152f) * cbf);
    b = _clamp8bit_full(yf + 1.8556f * cbf);
}

static void _ycbcr8_rgb888_bt709(uint8_t* in, int width, int height, uint8_t* out)
{
    for (int y = 0; y < height; y++) {
        for (int x = 0; x < width; x += 2) {
            uint32_t v = *((uint32_t*)&in[(y * width + x) * 2]);
            uint8_t cb = (int)((v >> 0) & 0xFF);
            uint8_t y1 = (v >> 8) & 0xFF;
            uint8_t cr = (int)((v >> 16) & 0xFF);
            uint8_t y2 = (v >> 24) & 0xFF;
            uint8_t r, g, b;

            _ycbcr888_rgb888_bt709(y1, cb, cr, r, g, b);
            out[(y * width + x + 0) * 3 + 0] = r;
            out[(y * width + x + 0) * 3 + 1] = g;
            out[(y * width + x + 0) * 3 + 2] = b;

            _ycbcr888_rgb888_bt709(y2, cb, cr, r, g, b);
            out[(y * width + x + 1) * 3 + 0] = r;
            out[(y * width + x + 1) * 3 + 1] = g;
            out[(y * width + x + 1) * 3 + 2] = b;
        }
    }
}

void convertVideoToRGB888(const v4l2_pix_format_mplane& fmt, uint8_t* in, uint8_t* out)
{
    int width = fmt.width, height = fmt.height;

    if (fmt.pixelformat == V4L2_PIX_FMT_UYVY)
        _ycbcr8_rgb888_bt709(in, width, height, out);
}
