#include "V4L2Device.hpp"

#include <cerrno>
#include <cstring>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <unistd.h>

using Buffer = V4L2Device::Buffer;
using BufferPtr = V4L2Device::BufferPtr;
static const int MAX_PLANES = 3;

Buffer::Buffer(int fd, int type, int index)
{
    m_fd = fd;
    m_mplane = (type == V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE)
        || (type == V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE);

    v4l2_plane planes[MAX_PLANES];

    m_v4l2buf.type = type;
    m_v4l2buf.memory = V4L2_MEMORY_MMAP;
    m_v4l2buf.index = index;

    if (m_mplane) {
        m_v4l2buf.m.planes = planes;
        m_v4l2buf.length = MAX_PLANES;
    }

    if (::ioctl(fd, VIDIOC_QUERYBUF, &m_v4l2buf))
        return; // Should not happen

    if (m_mplane) {
        m_v4l2planes = std::vector<v4l2_plane>(m_v4l2buf.length);
        std::copy(planes, planes + m_v4l2buf.length, m_v4l2planes.begin());
        m_v4l2buf.m.planes = m_v4l2planes.data();
        m_data = std::vector<uint8_t*>(m_v4l2buf.length, nullptr);
        m_length = std::vector<size_t>(m_v4l2buf.length, 0);
        for (int i = 0; i < m_v4l2buf.length; i++) {
            m_v4l2planes[i] = m_v4l2planes[i];
            m_data[i] = (uint8_t*)mmap(NULL, m_v4l2planes[i].length, PROT_READ | PROT_WRITE,
                MAP_SHARED, fd, m_v4l2planes[i].m.mem_offset);
            m_length[i] = m_v4l2planes[i].length;

            if (m_data[i] == MAP_FAILED) {
                m_length[i] = 0;
                m_data[i] = nullptr;
            }
        }
    } else {
        m_data = std::vector<uint8_t*>(1, nullptr);
        m_length = std::vector<size_t>(1, 0);
        m_data[0] = (uint8_t*)mmap(
            NULL, m_v4l2buf.length, PROT_READ | PROT_WRITE, MAP_SHARED, fd, m_v4l2buf.m.offset);
        m_length[0] = m_v4l2buf.length;
        if (m_data[0] == MAP_FAILED) {
            m_length[0] = 0;
            m_data[0] = nullptr;
        }
    }
}

Buffer::~Buffer()
{
    for (int i = 0; i < m_data.size(); i++) {
        if (!m_data[i] || !m_length[i])
            continue;
        munmap(m_data[i], m_length[i]);
        m_data[i] = nullptr;
        m_length[i] = 0;
    }
    m_data.clear();
    m_length.clear();
}

void Buffer::update(const v4l2_buffer& buf)
{
    if (m_mplane) {
        v4l2_plane* tmp = m_v4l2buf.m.planes;
        m_v4l2buf = buf;
        std::copy(buf.m.planes, buf.m.planes + buf.length, m_v4l2planes.begin());
        m_v4l2buf.m.planes = tmp;
    } else {
        m_v4l2buf = buf;
    }
}

int Buffer::numPlanes() const
{
    return m_mplane ? m_v4l2planes.size() : 1;
}

uint8_t* Buffer::data(unsigned int plane) const
{
    if (plane >= m_data.size())
        return nullptr;

    return m_data.at(plane);
}

size_t Buffer::length(unsigned int plane) const
{
    if (plane >= m_length.size())
        return 0;

    return m_length.at(plane);
}

size_t Buffer::bytesused(unsigned int plane) const
{
    if (m_mplane) {
        if (plane >= m_v4l2planes.size())
            return 0;

        return m_v4l2planes.at(plane).bytesused;
    } else {
        if (plane)
            return 0;
        return m_v4l2buf.bytesused;
    }
}

v4l2_buffer& Buffer::v4l2buf()
{
    return m_v4l2buf;
}

V4L2Device::V4L2Device(std::string& path)
{
    m_fd = ::open(path.c_str(), O_RDWR | O_NONBLOCK);
    if (m_fd == -1) {
        // Can't open device
        m_lastError = errno;
    } else {
        if (ioctl(VIDIOC_QUERYCAP, &m_caps)) {
            // Not V4L2 device
            ::close(m_fd);
            m_fd == -1;
        }
        m_mplane = (m_caps.device_caps & V4L2_CAP_VIDEO_CAPTURE_MPLANE)
            || (m_caps.device_caps & V4L2_CAP_VIDEO_OUTPUT_MPLANE);
    }
}

V4L2Device::~V4L2Device()
{
    if (m_fd != -1)
        ::close(m_fd);
}

bool V4L2Device::isOpen() const
{
    return m_fd != -1;
}

bool V4L2Device::isError() const
{
    return (m_fd == -1) || (m_lastError != 0);
}

int V4L2Device::error() const
{
    return m_lastError;
}

int V4L2Device::fd() const
{
    return m_fd;
}

std::string V4L2Device::errorString() const
{
    return std::string(strerror(m_lastError));
}

v4l2_buf_type V4L2Device::type() const
{
    if (m_caps.device_caps & V4L2_CAP_VIDEO_CAPTURE_MPLANE)
        return V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE;
    else if (m_caps.device_caps & V4L2_CAP_VIDEO_OUTPUT_MPLANE)
        return V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE;
    else if (m_caps.device_caps & V4L2_CAP_VBI_CAPTURE)
        return V4L2_BUF_TYPE_VBI_CAPTURE;
    else if (m_caps.device_caps & V4L2_CAP_VBI_OUTPUT)
        return V4L2_BUF_TYPE_VBI_OUTPUT;
    else
        return (v4l2_buf_type)0;
}

std::string V4L2Device::name() const
{
    return std::string((const char*)m_caps.card);
}

int V4L2Device::ioctl(int ctl, void* param)
{
    int result = ::ioctl(m_fd, ctl, param);
    m_lastError = (result < 0) ? errno : 0;
    return result;
}

void V4L2Device::requestBuffers(int count)
{
    v4l2_requestbuffers reqbuf;
    reqbuf.type = type();
    reqbuf.count = count;
    reqbuf.memory = V4L2_MEMORY_MMAP;

    if (ioctl(VIDIOC_REQBUFS, &reqbuf))
        return;

    m_buffers.resize(reqbuf.count);

    for (int i = 0; i < (int)reqbuf.count; i++)
        m_buffers[i] = std::shared_ptr<Buffer>(new Buffer(m_fd, reqbuf.type, i));
}

void V4L2Device::freeBuffers()
{
    m_buffers.clear();

    v4l2_requestbuffers reqbuf;
    reqbuf.type = type();
    reqbuf.count = 0;
    reqbuf.memory = V4L2_MEMORY_MMAP;

    ioctl(VIDIOC_REQBUFS, &reqbuf);
}

BufferPtr V4L2Device::buffer(int n) const
{
    if (n < 0 || n >= m_buffers.size())
        return nullptr;

    return m_buffers.at(n);
}

const std::vector<BufferPtr>& V4L2Device::buffers() const
{
    return m_buffers;
}

BufferPtr V4L2Device::dequeueBuffer()
{
    BufferPtr result = nullptr;

    v4l2_buffer v4l2buf;
    v4l2_plane planes[MAX_PLANES];

    v4l2buf.type = type();
    v4l2buf.memory = V4L2_MEMORY_MMAP;
    if (m_mplane) {
        v4l2buf.m.planes = planes;
        v4l2buf.length = MAX_PLANES;
    }
    v4l2buf.field = V4L2_FIELD_ANY;

    if (!ioctl(VIDIOC_DQBUF, &v4l2buf)) {
        if (v4l2buf.index < m_buffers.size()) {
            result = m_buffers[v4l2buf.index];
            if (result)
                result->update(v4l2buf);
        }
    }

    return result;
}

bool V4L2Device::queueBuffer(BufferPtr buffer)
{
    return !ioctl(VIDIOC_QBUF, &buffer->m_v4l2buf);
}

v4l2_format V4L2Device::getFormat()
{
    v4l2_format fmt;
    fmt.type = type();
    if (ioctl(VIDIOC_G_FMT, &fmt))
        return v4l2_format();
    return fmt;
}

v4l2_format V4L2Device::setFormat(const v4l2_format& fmt)
{
    v4l2_format newFmt = fmt;
    if (ioctl(VIDIOC_S_FMT, &newFmt))
        return getFormat();
    return newFmt;
}
