#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <thread>

/*!
 * \dir ./
 * This example shows how to enable genlock on board
 */

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device " << dev.name() << " is not an output device" << std::endl;
        return -1;
    }

    struct v4l2_control ctl;

    std::cout << "Setting genlock source to IN0" << std::endl;
    ctl.id = V4L2_CID_FORWARD_GENLOCK_SOURCE;
    ctl.value = V4L_FORWARD_GENLOCK_SRC_IN0;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl)) {
        std::cerr << "Cannot set genlock source: " << dev.errorString() << std::endl;
        return -1;
    }

    std::cout << "Waiting board to lock..." << std::endl;
    ctl.id = V4L2_CID_FORWARD_GENLOCK_STATE;

    int sec = 0;
    do {
        std::this_thread::sleep_for(std::chrono::milliseconds(1000));
        if (dev.ioctl(VIDIOC_G_CTRL, &ctl)) {
            std::cerr << "Cannot set genlock state: " << dev.errorString() << std::endl;
            return -1;
        }
        sec++;
    } while ((ctl.value == V4L_FORWARD_GENLOCK_LOCKING) && (sec < 30));

    std::cout << "Genlock state - ";
    switch (ctl.value) {
    case V4L_FORWARD_GENLOCK_MASTER:
        std::cout << "Master";
        break;
    case V4L_FORWARD_GENLOCK_NO_INPUT_SIGNAL:
        std::cout << "No signal";
        break;
    case V4L_FORWARD_GENLOCK_LOCKING:
        std::cout << "Locking";
        break;
    case V4L_FORWARD_GENLOCK_LOCKED:
        std::cout << "Locked";
        break;
    case V4L_FORWARD_GENLOCK_HOLDOVER:
        std::cout << "Holdover";
        break;
    default:
        std::cout << "Unknown";
        break;
    }
    std::cout << std::endl;

    return 0;
}
