#include "ANCParser.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-v4l2-ioctl.h"

#include <chrono>
#include <cmath>
#include <csignal>
#include <cstring>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <poll.h>

/*!
 * \dir ./
 * This example shows complex video+audio+vbi playback example
 */

namespace {
static const int BUFFER_COUNT = 8;
static const int AUDIO_SAMPLE_RATE = 48000;
static const int AUDIO_CHANNELS = 4; // 4, 8, 16
static const int TEST_PERIOD_FIELDS = 10;
}

static bool shouldStop = false;
static uint8_t* videoTest;
int audioBufSize[2];
int32_t* audioBufSilence;
int32_t* audioBufSin;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

// VBI should be explicit enabled by controls
void toggleVBI(V4L2Device& dev, bool on)
{
    struct v4l2_control ctl;
    ctl.id = V4L2_CID_FORWARD_ENABLE_VANC;
    ctl.value = on;

    if (dev.ioctl(VIDIOC_S_CTRL, &ctl))
        std::cerr << "Cannot toggle VANC: " << dev.errorString() << std::endl;
}

v4l2_format setupFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Each field are in separate buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;

    // 8-bit
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_COMPLEX_8BIT;
    // 10-bit
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_COMPLEX_10BIT;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Playback frame format is " << fmt
              << " plane sizes: " << fmt.fmt.pix_mp.plane_fmt[0].sizeimage << " "
              << fmt.fmt.pix_mp.plane_fmt[1].sizeimage << " "
              << fmt.fmt.pix_mp.plane_fmt[2].sizeimage << std::endl;

    return fmt;
}

int getSamplesPerField(V4L2Device& dev, bool field)
{
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_G_DV_TIMINGS, &tim)) {
        std::cerr << "Cannot get device timigs: " << dev.errorString() << std::endl;
        return 0;
    }
    int totalwidth = tim.bt.hsync + tim.bt.hbackporch + tim.bt.width + tim.bt.hfrontporch;
    int totallines[2];
    if (tim.bt.interlaced) {
        totallines[0]
            = tim.bt.vsync + tim.bt.vbackporch + tim.bt.height / 2 + tim.bt.il_vfrontporch;
        totallines[1]
            = tim.bt.il_vsync + tim.bt.il_vbackporch + tim.bt.height / 2 + tim.bt.vfrontporch;
    } else {
        totallines[0] = tim.bt.vsync + tim.bt.vbackporch + tim.bt.height + tim.bt.vfrontporch;
        totallines[1] = 0;
    }
    double frame_samples = AUDIO_SAMPLE_RATE
        * uint64_t((totallines[0] + totallines[1]) * totalwidth) / tim.bt.pixelclock;
    if (tim.bt.flags & V4L2_DV_FL_REDUCED_FPS)
        frame_samples = frame_samples * 1001.0 / 1000.0;
    double line_samples = frame_samples / double(totallines[0] + totallines[1]);

    int odd_samples = round(line_samples * totallines[0]);
    int even_samples = frame_samples - odd_samples;

    return (field && tim.bt.interlaced) ? even_samples : odd_samples;
}

void preroll(V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers)
{
    // Fill playback queue
    for (auto buf : buffers) {
        int field = buf->v4l2buf().index % 2;
        buf->v4l2buf().field = field ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
        buf->v4l2buf().m.planes[1].bytesused = audioBufSize[field] * AUDIO_CHANNELS * 4;
        dev.queueBuffer(buf);
    }
}

void mainLoop(V4L2Device& dev, const v4l2_pix_format_mplane& videoFormat)
{
    int type = dev.type();

    // Start capture
    dev.ioctl(VIDIOC_STREAMON, &type);

    pollfd pollfd;
    pollfd.fd = dev.fd();
    pollfd.events = POLLOUT;

    bool bottomField = false;
    int num = 0;

    while (!shouldStop) {
        // Wait for frame
        poll(&pollfd, 1, 100);

        if (pollfd.revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Timeout, no frame
        if (!(pollfd.revents & POLLOUT))
            continue;

        // Get buffer from queue
        V4L2Device::BufferPtr b = dev.dequeueBuffer();

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);

        std::cout << "Sended buffer at "
                  << (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec
                  << "ns, id = " << b->v4l2buf().index << ", sequence = " << b->v4l2buf().sequence
                  << ", field = " << b->v4l2buf().field << ", timestamp = "
                  << (uint64_t)b->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + b->v4l2buf().timestamp.tv_usec * 1000
                  << "ns, video size = " << b->v4l2buf().m.planes[0].bytesused
                  << ", anc size = " << b->v4l2buf().m.planes[1].bytesused
                  << ", meta size = " << b->v4l2buf().m.planes[2].bytesused << std::endl;

        uint8_t* videoData = b->data(0);
        int32_t* audioData = (int32_t*)b->data(1);

        b->v4l2buf().field = (videoFormat.field == V4L2_FIELD_NONE)
            ? V4L2_FIELD_NONE
            : (bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP);

        int n = num % (TEST_PERIOD_FIELDS);
        uint8_t nY = (uint8_t)(16 + n * 235 / TEST_PERIOD_FIELDS);
        for (int i = 0; i < b->length() / 2; i++) {
            videoData[i * 2 + 0] = 0x80;
            videoData[i * 2 + 1] = nY;
        }
        memcpy(audioData, ((num % TEST_PERIOD_FIELDS) == 0) ? audioBufSin : audioBufSilence,
            audioBufSize[bottomField ? 1 : 0] * AUDIO_CHANNELS * 4);
        b->v4l2buf().m.planes[1].bytesused = audioBufSize[bottomField ? 1 : 0] * AUDIO_CHANNELS * 4;

        const struct forward_v4l2_complex_info* info
            = (struct forward_v4l2_complex_info*)b->data(2);
        std::cout << "Complex info seq = " << info->seq << ", field = " << info->field
                  << ", timestamp = " << info->timestamp
                  << ", hw_timestamp = " << info->hw_timestamp << std::endl;

        // Return buffer back to queue
        dev.queueBuffer(b);
        bottomField = !bottomField;
        num++;
    }

    // Stop capture
    dev.ioctl(VIDIOC_STREAMOFF, &type);
}

int main(int argc, char** argv)
{
    std::string devPath;
    if (argc != 2) {
        std::cout << "Usage: " << argv[0] << " device" << std::endl;
        return 0;
    }

    devPath = argv[1];
    std::cout << "Openning " << devPath << std::endl;
    V4L2Device dev(devPath);

    if (!dev.isOpen()) {
        std::cerr << "Cannot open device: " << dev.errorString() << std::endl;
        return -1;
    }

    if (dev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device (" << dev.name() << ") is not an output device" << std::endl;
        return -1;
    }

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format format;

    toggleVBI(dev, true);
    format = setupFormat(dev);
    dev.requestBuffers(BUFFER_COUNT);

    // Alloc and fill audio buffer
    audioBufSize[0] = getSamplesPerField(dev, false);
    audioBufSize[1] = getSamplesPerField(dev, true);
    int bufSize = std::max(audioBufSize[0], audioBufSize[1]);
    audioBufSilence = new int32_t[bufSize * AUDIO_CHANNELS];
    audioBufSin = new int32_t[bufSize * AUDIO_CHANNELS];

    memset(audioBufSin, 0, bufSize * AUDIO_CHANNELS * 4);
    memset(audioBufSilence, 0, bufSize * AUDIO_CHANNELS * 4);

    for (int i = 0; i < 96; i++) {
        float v = (float)sin(M_PI * (i + 1) / 48);
        audioBufSin[i * AUDIO_CHANNELS + 0] = v * (float)((1U << 23) - 1);
        audioBufSin[i * AUDIO_CHANNELS + 1] = v * (float)((1U << 23) - 1);
    }

    std::cout << "Playback with " << dev.buffers().size() << " buffers" << std::endl;

    preroll(dev, dev.buffers());
    mainLoop(dev, format.fmt.pix_mp);

    dev.freeBuffers();

    return 0;
}
