#include "ALSADevice.hpp"
#include "Board.hpp"
#include "Utils.hpp"
#include "V4L2Device.hpp"
#include "forward-alsa-ioctl.h"
#include "forward-v4l2-ioctl.h"

#include <arpa/inet.h>
#include <climits>
#include <cmath>
#include <csignal>
#include <cstring>
#include <iomanip>
#include <iostream>
#include <list>
#include <net/if.h>
#include <poll.h>
#include <regex>
#include <sys/ioctl.h>
#include <tuple>

/*!
 * \dir ./
 * This example shows video + audio playback example on FD2110
 */

namespace {
static const int BUFFER_COUNT = 8;
// ALSA chunk size, should be less than frame (field) period
static const int AUDIO_PERIOD_SIZE_US = 10000;
static const int AUDIO_BUFFER_SIZE_US = 1000000;
static const int AUDIO_CHANNELS = 2;
static const snd_pcm_uframes_t AUDIO_BUFFER_SIZE = 2048 * 16;
static const snd_pcm_uframes_t AUDIO_SAMPLE_RATE = 48000;
}

static bool shouldStop = false;
static uint8_t* colorbar;
static uint8_t* white;
snd_pcm_uframes_t audioBufSize;
int32_t* audioBufSilence;
int32_t* audioBufSin;

void stopSignal(int sig)
{
    if ((sig == SIGTERM) || (sig == SIGINT))
        shouldStop = true;
}

bool str2ipv6(const std::string& str, uint8_t* ip, bool& v6)
{
    struct in_addr addr;
    struct in6_addr addr6;

    memset(ip, 0, 16);

    if (inet_pton(AF_INET, str.c_str(), &addr)) {
        uint8_t* in_p = (uint8_t*)&addr.s_addr;

        for (int i = 0; i < 4; i++)
            ip[12 + i] = in_p[i];
        ip[11] = 0xFF;
        ip[10] = 0xFF;
        v6 = false;
    } else if (inet_pton(AF_INET6, str.c_str(), &addr6)) {
        for (int i = 0; i < 16; i++)
            ip[i] = addr6.s6_addr[i];
        v6 = true;
    } else
        return false;

    return true;
}

int setupVideoStream(V4L2Device& dev, const std::string& videoAddr, uint16_t videoPort,
    uint16_t videoSrcPort = 20000)
{
    struct v4l2_forward_stream_address v4l2addr;
    bool v6;

    if (!str2ipv6(videoAddr, v4l2addr.ipv6, v6)) {
        std::cerr << "Cannot parse video IP" << std::endl;
        return -EINVAL;
    }
    v4l2addr.port = videoPort;

    struct v4l2_ext_control ex_ctl;
    struct v4l2_ext_controls ex_ctls;
    struct v4l2_control ctl;

    // Setup IP + Port
    memset(&ex_ctl, 0, sizeof(ex_ctl));
    memset(&ex_ctls, 0, sizeof(ex_ctls));
    memset(&ctl, 0, sizeof(ctl));

    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_SOURCE_PORT;
    ctl.value = videoSrcPort;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video source port: " << dev.errorString() << std::endl;
        return dev.error();
    }

    ex_ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_ADDRESS;
    ex_ctl.size = sizeof(v4l2addr);
    ex_ctl.ptr = &v4l2addr;
    ex_ctls.which = V4L2_CTRL_WHICH_CUR_VAL;
    ex_ctls.count = 1;
    ex_ctls.controls = &ex_ctl;

    if (dev.ioctl(VIDIOC_S_EXT_CTRLS, &ex_ctls) < 0) {
        std::cerr << "Cannot set video stream address: " << dev.errorString() << std::endl;
        return dev.error();
    }

    // Setup output stream color mode + bit depth - should be same as in SDP.
    // Board can do some conversion between stream and v4l2 device format (e.g. 8->10 bit
    //   conversion).
    // Set YCbCr 4:2:2, 10-bit - default format for 2110->SDI converters like AJA.
    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_COLOR;
    ctl.value = V4L_FORWARD_FD2110_STREAM_COLOR_YCBCR422;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream color format: " << dev.errorString() << std::endl;
        return dev.error();
    }

    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_BPC;
    ctl.value = V4L_FORWARD_FD2110_STREAM_BPC_10BIT;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream bpc: " << dev.errorString() << std::endl;
        return dev.error();
    }

    ctl.id = V4L2_CID_FORWARD_FD2110_STREAM_SSRC;
    ctl.value = 1000;
    if (dev.ioctl(VIDIOC_S_CTRL, &ctl) < 0) {
        std::cerr << "Cannot set video stream SSRC: " << dev.errorString() << std::endl;
        return dev.error();
    }

    return 0;
}

int alsaControlLookup(
    snd_ctl_t* dev, snd_ctl_elem_id_t* id, const std::string& name, int io, int stream)
{
    snd_ctl_elem_info_t* info;
    snd_ctl_elem_info_alloca(&info);

    snd_ctl_elem_id_set_interface(id, SND_CTL_ELEM_IFACE_PCM);
    snd_ctl_elem_id_set_device(id, io);
    snd_ctl_elem_id_set_subdevice(id, stream);
    snd_ctl_elem_id_set_name(id, name.c_str());

    snd_ctl_elem_info_set_id(info, id);
    int result = snd_ctl_elem_info(dev, info);
    if (result) {
        std::cerr << "Can't find ALSA control \"" << name << "\": " << snd_strerror(result)
                  << std::endl;
        return result;
    }
    snd_ctl_elem_info_get_id(info, id);

    return 0;
}

int alsaControlWriteArray(
    snd_ctl_t* dev, int io, int stream, const std::string& name, const uint8_t* data, size_t size)
{
    snd_ctl_elem_value_t* value;
    snd_ctl_elem_value_alloca(&value);
    snd_ctl_elem_id_t* id;
    snd_ctl_elem_id_alloca(&id);
    int result;

    result = alsaControlLookup(dev, id, name, io, stream);
    if (result)
        return result;

    snd_ctl_elem_value_set_id(value, id);
    for (size_t i = 0; i < size; i++)
        snd_ctl_elem_value_set_byte(value, i, data[i]);
    result = snd_ctl_elem_write(dev, value);
    if (result) {
        std::cerr << "Cannot write ALSA control: " << snd_strerror(result) << std::endl;
        return result;
    }
    return result;
}

int alsaControlWriteEnum(snd_ctl_t* dev, int io, int stream, const std::string& name, int v)
{
    snd_ctl_elem_value_t* value;
    snd_ctl_elem_value_alloca(&value);
    snd_ctl_elem_id_t* id;
    snd_ctl_elem_id_alloca(&id);
    int result;

    result = alsaControlLookup(dev, id, name, io, stream);
    if (result)
        return result;

    snd_ctl_elem_value_set_id(value, id);
    snd_ctl_elem_value_set_enumerated(value, 0, v);
    result = snd_ctl_elem_write(dev, value);
    if (result) {
        std::cerr << "Cannot write ALSA control: " << snd_strerror(result) << std::endl;
        return result;
    }
    return result;
}

int alsaControlWriteInt(snd_ctl_t* dev, int io, int stream, const std::string& name, int v)
{
    snd_ctl_elem_value_t* value;
    snd_ctl_elem_value_alloca(&value);
    snd_ctl_elem_id_t* id;
    snd_ctl_elem_id_alloca(&id);
    int result;

    result = alsaControlLookup(dev, id, name, io, stream);
    if (result)
        return result;

    snd_ctl_elem_value_set_id(value, id);
    snd_ctl_elem_value_set_integer(value, 0, v);
    result = snd_ctl_elem_write(dev, value);
    if (result) {
        std::cerr << "Cannot write ALSA control: " << snd_strerror(result) << std::endl;
        return result;
    }
    return result;
}

int setupAudioStream(ALSADevice& dev, const std::string& audioAddr, uint16_t audioPort,
    uint8_t audioPT, uint16_t audioSrcPort = 20001)
{
    int result;
    struct snd_forward_stream_address addr;
    bool v6;

    if (!str2ipv6(audioAddr, addr.ipv6, v6)) {
        std::cerr << "Cannot parse audio IP" << std::endl;
        return -EINVAL;
    }
    addr.port = audioPort;
    addr.payload_type = audioPT;

    // Get sound card and io number from PCM device
    snd_pcm_info_t* pcm_info;
    snd_pcm_info_alloca(&pcm_info);
    result = snd_pcm_info(dev.pcm().get(), pcm_info);
    if (result) {
        std::cerr << "Cannot get ALSA PCM info: " << snd_strerror(result) << std::endl;
        return result;
    }

    int card = snd_pcm_info_get_card(pcm_info);
    int io = snd_pcm_info_get_device(pcm_info);

    // Open control device
    snd_ctl_t* ctl;
    std::string ctl_name = std::string("hw:") + std::to_string(card);
    snd_ctl_open(&ctl, ctl_name.c_str(), 0);

    // Set stream address control for device
    result = alsaControlWriteArray(ctl, io, SND_PCM_STREAM_PLAYBACK, "PCM Playback Stream Address",
        (uint8_t*)&addr, sizeof(addr));
    if (result)
        return result;

    result = alsaControlWriteInt(
        ctl, io, SND_PCM_STREAM_PLAYBACK, "PCM Playback Stream Source Port", audioSrcPort);
    if (result)
        return result;

    // Set stream format - L24
    result = alsaControlWriteEnum(ctl, io, SND_PCM_STREAM_PLAYBACK, "PCM Playback Stream Format",
        SND_FORWARD_STREAM_FORMAT_L24);
    if (result)
        return result;

    // Set stream period - 128us
    result = alsaControlWriteEnum(ctl, io, SND_PCM_STREAM_PLAYBACK, "PCM Playback Stream Period",
        SND_FORWARD_STREAM_PERIOD_125US);
    if (result)
        return result;

    // Set stream number of channels - 4 = one SDI group
    result
        = alsaControlWriteInt(ctl, io, SND_PCM_STREAM_PLAYBACK, "PCM Playback Stream Channels", 4);
    if (result)
        return result;

    // Set stream SSRC
    result
        = alsaControlWriteInt(ctl, io, SND_PCM_STREAM_PLAYBACK, "PCM Playback Stream SSRC", 1001);
    if (result)
        return result;

    snd_ctl_close(ctl);

    return 0;
}

int setupStream(const std::string& board, int stream, const std::string& videoAddr,
    uint16_t videoPort, const std::string& audioAddr, uint16_t audioPort, uint8_t audioPT,
    std::string& videoDev, std::string& audioDev)
{
    Board::Info info = Board::fromPath(board);
    int result = 0;

    if (info.type == Board::Info::Unknown) {
        std::cerr << "Wrong board path, should be /dev/forward/fd2110-*" << std::endl;
        return -ENODEV;
    } else if (info.type != Board::Info::FD2110) {
        std::cerr << "Not a FD2110 board" << std::endl;
        return -EINVAL;
    }

    Board fd2110(info);

    // Enable board stream
    std::cout << "Enabling output stream #" << stream << "..." << std::endl;
    result = fd2110.switchIOMode(stream + 16 + 2, Board::IOMode::Output);
    if (result) {
        std::cerr << "Cannot enable stream #" << stream << ": " << strerror(result) << std::endl;
        return result;
    }

    // Getting devices
    info = fd2110.info();

    videoDev = info.videoDevs[stream + 16 + 2];
    audioDev = info.audioDevs[stream + 16 + 2];

    std::cout << "Stream #" << stream << " enabled, video device: " << videoDev
              << ", audio device: " << audioDev << std::endl;

    // Setup V4L2 device - pass stream address and format to driver
    V4L2Device video(videoDev);
    if (!video.isOpen()) {
        std::cerr << "Cannot open video device: " << video.errorString() << std::endl;
        return -1;
    }
    std::cout << "Configuring V4L2 device..." << std::endl;
    result = setupVideoStream(video, videoAddr, videoPort);
    if (result)
        return result;

    // Setup ALSA device - pass stream address, payload type and format to driver
    ALSADevice audio(audioDev, SND_PCM_STREAM_PLAYBACK);
    if (!audio.isOpen()) {
        std::cerr << "Cannot open audio device: " << audio.errorString() << std::endl;
        return -1;
    }
    std::cout << "Configuring ALSA device..." << std::endl;
    result = setupAudioStream(audio, audioAddr, audioPort, audioPT);
    if (result)
        return result;

    return 0;
}

//! Get required audio samples per video frame
snd_pcm_uframes_t getAudioPerVideoFrame(V4L2Device& dev, const v4l2_pix_format_mplane& fmt)
{
    v4l2_dv_timings tim;
    if (dev.ioctl(VIDIOC_G_DV_TIMINGS, &tim))
        return 0;

    uint64_t denom = tim.bt.pixelclock;
    uint64_t num
        = AUDIO_SAMPLE_RATE * V4L2_DV_BT_FRAME_WIDTH(&tim.bt) * V4L2_DV_BT_FRAME_HEIGHT(&tim.bt);

    // FIXME: No /1.001 compensation
    if (tim.bt.interlaced && (fmt.field == V4L2_FIELD_ALTERNATE))
        denom *= 2;

    return num / denom;
}

v4l2_format setupVideoFormat(V4L2Device& dev)
{
    // Get current frame format
    v4l2_format fmt = dev.getFormat();

    // Set field interleave - both fields are in one buffer
    fmt.fmt.pix_mp.field = V4L2_FIELD_INTERLACED;
    // Each field are in separate buffer
    // fmt.fmt.pix_mp.field = V4L2_FIELD_ALTERNATE;
    // 8-bit UYVY
    fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_UYVY;
    // 8-bit YUYV
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_YUYV;
    // 10-bit v210
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_V210;
    // Raw - whole SDI frame, with HANC, VANC, 10-bit dense packed
    // fmt.fmt.pix_mp.pixelformat = V4L2_PIX_FMT_SL_RAW;

    // Set and adjust format
    fmt = dev.setFormat(fmt);

    std::cout << "Playback frame format is " << fmt << std::endl;

    return fmt;
}

void preroll(
    V4L2Device& dev, const std::vector<V4L2Device::BufferPtr>& buffers, v4l2_pix_format_mplane fmt)
{
    // Fill capture queue
    for (auto buf : buffers) {
        if (fmt.field == V4L2_FIELD_ALTERNATE)
            buf->v4l2buf().field = (buf->v4l2buf().index % 2) ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;

        if (dev.type() == V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE)
            memcpy(buf->data(), white, fmt.plane_fmt[0].sizeimage);

        dev.queueBuffer(buf);
    }
}

void prerollAudio(ALSADevice& dev, int audioBufSize, int buffers)
{
    int32_t* audio_buf = new int32_t[audioBufSize * AUDIO_CHANNELS];
    memset(audio_buf, 0, audioBufSize * AUDIO_CHANNELS * sizeof(int32_t));

    for (int i = 0; i < buffers; i++)
        snd_pcm_writei(dev.pcm().get(), audio_buf, audioBufSize);

    delete[] audio_buf;
}

void setupAudio(ALSADevice& dev)
{
    snd_pcm_hw_params_t* params = nullptr;
    snd_pcm_sw_params_t* sw_params = nullptr;
    snd_pcm_format_t format;
    snd_pcm_uframes_t period;
    unsigned int channels, rate;
    int dir;

    // Alloc hardware parameter structs
    snd_pcm_hw_params_malloc(&params);

    // Get default hardware parameters
    snd_pcm_hw_params_any(dev.pcm().get(), params);

    // Interleave channels
    snd_pcm_hw_params_set_access(dev.pcm().get(), params, SND_PCM_ACCESS_RW_INTERLEAVED);

    // Set period and buffer sizes
    snd_pcm_hw_params_set_period_time(dev.pcm().get(), params, AUDIO_PERIOD_SIZE_US, 0);
    snd_pcm_hw_params_set_buffer_time(dev.pcm().get(), params, AUDIO_BUFFER_SIZE_US, 0);

    // Set number of channels
    snd_pcm_hw_params_set_channels(dev.pcm().get(), params, AUDIO_CHANNELS);

    // Set hardware parameters, you can also change format, channels and other params prior this
    // call
    snd_pcm_hw_params(dev.pcm().get(), params);

    // Extract accepted parameters,
    snd_pcm_hw_params_get_format(params, &format);
    snd_pcm_hw_params_get_channels(params, &channels);
    snd_pcm_hw_params_get_rate(params, &rate, &dir);
    snd_pcm_hw_params_get_period_size(params, &period, &dir);

    snd_pcm_hw_params_free(params);

    // Alloc software parameter structs
    snd_pcm_sw_params_alloca(&sw_params);
    // Get current software parameters
    snd_pcm_sw_params_current(dev.pcm().get(), sw_params);
    // Enable ALSA timestamping so we can always sync audio with video
    snd_pcm_sw_params_set_tstamp_mode(dev.pcm().get(), sw_params, SND_PCM_TSTAMP_ENABLE);
    // Disable ALSA auto-start
    snd_pcm_sw_params_set_start_threshold(dev.pcm().get(), sw_params, LONG_MAX);
    // Set software parameters
    snd_pcm_sw_params(dev.pcm().get(), sw_params);

    snd_pcm_prepare(dev.pcm().get());

    std::cout << "ALSA playback " << channels << " x " << snd_pcm_format_name(format) << " x "
              << rate << " with period of " << period << " samples" << std::endl;
}

void mainLoop(V4L2Device& videoDev, ALSADevice& audioDev, const v4l2_pix_format_mplane& videoFormat)
{
    int videoType = videoDev.type();

    bool bottomField = false;
    int num = 0;

    // Alloc and setup ALSA status structure - used to get current ALSA status
    std::shared_ptr<snd_pcm_status_t> status;
    snd_pcm_status_t* sp;
    snd_pcm_status_malloc(&sp);
    status = std::shared_ptr<snd_pcm_status_t>(sp, snd_pcm_status_free);

    // Forward-specific setup - this will tell ALSA and driver to return real timestamp in status
    // This timestamp will be same as in corresponding video buffer
    snd_pcm_audio_tstamp_config_t config;
    config.report_delay = 0;
    config.type_requested = SND_PCM_AUDIO_TSTAMP_TYPE_LINK_ABSOLUTE;
    snd_pcm_status_set_audio_htstamp_config(status.get(), &config);

    // Start playback
    videoDev.ioctl(VIDIOC_STREAMON, &videoType);
    // Start audio playback
    snd_pcm_start(audioDev.pcm().get());

    pollfd pollfd[1];

    pollfd[0].fd = videoDev.fd();
    pollfd[0].events = POLLOUT;

    while (!shouldStop) {
        // Wait for frame
        poll(pollfd, 1, 100);

        timespec timestamp;
        clock_gettime(CLOCK_MONOTONIC, &timestamp);
        uint64_t curTs = (uint64_t)timestamp.tv_sec * 1000000000ULL + timestamp.tv_nsec;

        if (pollfd[0].revents & (POLLERR | POLLHUP | POLLNVAL))
            break;

        // Video/VBI frame sended
        if (pollfd[0].revents & POLLOUT) {
            // Get buffer from driver, video first.
            // Video and VBI is always in sync if number of queued buffers are always same
            // To be sure, you can check sequence fields
            auto videoBuf = videoDev.dequeueBuffer();

            uint64_t vidTs = (uint64_t)videoBuf->v4l2buf().timestamp.tv_sec * 1000000000ULL
                + videoBuf->v4l2buf().timestamp.tv_usec * 1000;

            std::cout << "Sended video buffer at " << curTs << "ns, "
                      << "video id = " << videoBuf->v4l2buf().index
                      << ", video sequence = " << videoBuf->v4l2buf().sequence
                      << ", video field = " << videoBuf->v4l2buf().field
                      << ", video timestamp = " << vidTs
                      << "ns, video size = " << videoBuf->v4l2buf().bytesused << std::endl;

            // Get current ALSA status
            snd_pcm_status(audioDev.pcm().get(), status.get());
            // .. and timestamp
            snd_htimestamp_t ts;
            snd_pcm_status_get_audio_htstamp(status.get(), &ts);
            // Timestamp, drop ns part as in V4L2
            uint64_t audioTs = (uint64_t)ts.tv_sec * 1000000000ULL + (ts.tv_nsec / 1000) * 1000;
            snd_pcm_sframes_t audioBuffered = snd_pcm_status_get_delay(status.get());

            /*
             *  PREPARE DATA HERE
             */

            memcpy(videoBuf->data(), ((num % 8) < 1) ? colorbar : white,
                videoFormat.plane_fmt[0].sizeimage);

            // Don't forget to set correct field for V4L2_FIELD_ALTERNATE mode
            if (videoFormat.field == V4L2_FIELD_ALTERNATE) {
                videoBuf->v4l2buf().field = bottomField ? V4L2_FIELD_BOTTOM : V4L2_FIELD_TOP;
                bottomField = !bottomField;
            }

            snd_pcm_sframes_t size = snd_pcm_writei(audioDev.pcm().get(),
                ((num % 8) < 1) ? audioBufSin : audioBufSilence, audioBufSize);

            std::cout << "Sending " << size << " audio frames at " << curTs << "ns, "
                      << "audio timestamp = " << audioTs << ", buffered = " << audioBuffered
                      << std::endl;

            // Return video/vbi buffers back to queue
            videoDev.queueBuffer(videoBuf);

            num++;
        }
    }

    // Stop audio playback
    snd_pcm_drop(audioDev.pcm().get());
    // Stop playback
    videoDev.ioctl(VIDIOC_STREAMOFF, &videoType);
}

int main(int argc, char** argv)
{
    std::string board, videoAddr, audioAddr;
    int stream;
    int ethernet;
    uint8_t audioPT;
    uint16_t videoPort, audioPort;
    std::string videoPath, audioPath;

    if (argc != 8) {
        std::cout << "Usage: " << argv[0]
                  << " board stream_idx video_addr video_port audio_addr audio_port audio_pt"
                  << std::endl;
        return 0;
    }

    // FD2110 has (16 input streams + 16 output streams) x 2 Ethernets = 64 streams total
    // Streams represented as I/O pin, in order: 2x SDI, 16x eth0 inputs, 16x eth0 outputs, 16x eth1
    // inputs, 16x eth1 outputs
    board = argv[1];
    stream = std::stoi(argv[2]);
    videoAddr = argv[3];
    videoPort = std::stoul(argv[4]);
    audioAddr = argv[5];
    audioPort = std::stoul(argv[6]);
    audioPT = std::stoi(argv[7]);

    // Setup stream parameters - addresses, ports, payload type, pixel format, audio format, etc.
    int result = setupStream(
        board, stream, videoAddr, videoPort, audioAddr, audioPort, audioPT, videoPath, audioPath);

    // Common playback code below...
    std::cout << "Openning " << videoPath << std::endl;
    V4L2Device videoDev(videoPath);

    if (!videoDev.isOpen()) {
        std::cerr << "Cannot open video device: " << videoDev.errorString() << std::endl;
        return -1;
    }

    if (videoDev.type() != V4L2_BUF_TYPE_VIDEO_OUTPUT_MPLANE) {
        std::cerr << "Device " << videoPath << " is not an output video device" << std::endl;
        return -1;
    }

    ALSADevice audioDev(audioPath, SND_PCM_STREAM_PLAYBACK);

    if (!audioDev.isOpen()) {
        std::cerr << "Cannot open audio device: " << audioDev.errorString() << std::endl;
        return -1;
    }

    setupAudio(audioDev);

    signal(SIGTERM, &stopSignal);
    signal(SIGINT, &stopSignal);

    v4l2_format videoFormat;

    videoFormat = setupVideoFormat(videoDev);

    videoDev.requestBuffers(BUFFER_COUNT);

    colorbar = new uint8_t[videoFormat.fmt.pix_mp.plane_fmt[0].sizeimage];
    generateRP219ColorBarUYVY(colorbar, videoFormat.fmt.pix_mp.pixelformat,
        videoFormat.fmt.pix_mp.width, videoFormat.fmt.pix_mp.height);
    white = new uint8_t[videoFormat.fmt.pix_mp.plane_fmt[0].sizeimage];
    for (int i = 0; i < videoFormat.fmt.pix_mp.plane_fmt[0].sizeimage / 4; i++)
        ((uint32_t*)white)[i] = 0xF080F080;

    // Alloc audio buffer
    audioBufSize = getAudioPerVideoFrame(videoDev, videoFormat.fmt.pix_mp);
    audioBufSilence = new int32_t[audioBufSize * AUDIO_CHANNELS];
    audioBufSin = new int32_t[audioBufSize * AUDIO_CHANNELS];
    for (snd_pcm_uframes_t i = 0; i < audioBufSize; i++) {
        audioBufSilence[i * 2 + 0] = 0;
        audioBufSilence[i * 2 + 1] = 0;
        audioBufSin[i * 2 + 0]
            = sin((float)i / (audioBufSize / 10.0f) * 2.0f * M_PI) * (float)(1 << 30);
        audioBufSin[i * 2 + 1]
            = sin((float)i / (audioBufSize / 10.0f) * 2.0f * M_PI) * (float)(1 << 30);
    }

    std::cout << "Playback with " << videoDev.buffers().size() << " buffers" << std::endl;

    // Preroll video
    preroll(videoDev, videoDev.buffers(), videoFormat.fmt.pix_mp);
    prerollAudio(audioDev, getAudioPerVideoFrame(videoDev, videoFormat.fmt.pix_mp),
        videoDev.buffers().size());

    mainLoop(videoDev, audioDev, videoFormat.fmt.pix_mp);

    delete[] audioBufSilence;
    delete[] audioBufSin;
    delete[] colorbar;
    delete[] white;

    videoDev.freeBuffers();

    return 0;
}
